<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Backend\FrameStyles;
use Spatie\Dropbox\Client;
use League\Flysystem\Filesystem;
use Spatie\FlysystemDropbox\DropboxAdapter;
use App\Services\CronService;
use Carbon\Carbon;
use ZipArchive;

class ExportFramesCsv extends Command
{
    protected $signature = 'exportFrameCsv:run';
    protected $description = 'Export all updated frames data to compressed CSV and upload to Dropbox';
    
    protected $dbChunkSize = 1000;
    protected $maxRetries = 3;
    protected $retryDelay = 5;
    
    public function handle()
    {
        try {
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '-1');
            
            $chunkSize = 50000;
            $tempDir = storage_path('app/frames_export_temp/');
            $currentDate = date('Y-m-d');
            
            // Create temp directory
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }
            
            // 1. Generate chunked CSVs
            $this->info("Starting chunked CSV generation...");
            $chunkFiles = $this->generateChunkedCsvs($tempDir, $chunkSize);
            
            // 2. Upload chunks to Dropbox
            $this->info("Uploading chunked CSVs to Dropbox...");
            $uploadedPaths = $this->uploadChunksToDropbox($chunkFiles, $currentDate);
            
            // 3. Merge in Dropbox
            $this->info("Merging CSVs in Dropbox...");
            $finalPath = $this->mergeDropboxCsvs($uploadedPaths, $currentDate);
            
            // 4. Clean up
            $this->cleanLocalFiles($chunkFiles, $tempDir);
            
            $this->info("Process completed successfully. Final file: ".$finalPath);
            CronService::log($this->signature, '1', "Export completed");
            
        } catch (\Exception $e) {
            CronService::log($this->signature, '0', $e->getMessage());
            $this->error('Error: ' . $e->getMessage());
        }
    }

    protected function cleanLocalFiles(array $files, string $directory)
    {
        // Delete individual chunk files
        foreach ($files as $file) {
            if (file_exists($file)) {
                unlink($file);
                $this->info("Deleted local file: " . basename($file));
            }
        }
        
        // Delete the temp directory
        if (is_dir($directory)) {
            @rmdir($directory);
            $this->info("Deleted temporary directory: " . basename($directory));
        }
    }
    
    protected function generateChunkedCsvs($directory, $chunkSize)
    {
        $headers = $this->getCsvHeaders();
        $chunkFiles = [];
        $currentChunk = 0;
        $processedRecords = 0;
        $totalRecords = FrameStyles::count();
        
        // Open first file
        $currentFile = $this->openNewChunkFile($directory, $currentChunk);
        fputcsv($currentFile, $headers);
        $chunkFiles[] = stream_get_meta_data($currentFile)['uri'];
        
        FrameStyles::chunk($chunkSize, function($styles) use (&$currentFile, &$currentChunk, &$chunkFiles, $directory, $headers, &$processedRecords, $totalRecords) {
            foreach ($styles as $style) {
                $row = $this->formatStyleRow($style, $headers);
                fputcsv($currentFile, $row);
                
                $processedRecords++;
                
                if ($processedRecords % 10000 === 0) {
                    $progress = round(($processedRecords / $totalRecords) * 100, 2);
                    $this->info("Processed {$processedRecords} of {$totalRecords} records ({$progress}%)");
                }
            }
            
            // Close current file and open new one for next chunk
            fclose($currentFile);
            $currentChunk++;
            $currentFile = $this->openNewChunkFile($directory, $currentChunk);
            fputcsv($currentFile, $headers);
            $chunkFiles[] = stream_get_meta_data($currentFile)['uri'];
        });
        
        fclose($currentFile);
        return $chunkFiles;
    }
    
    protected function openNewChunkFile($directory, $chunkNumber)
    {
        $path = $directory . 'frames_chunk_' . str_pad($chunkNumber, 4, '0', STR_PAD_LEFT) . '.csv';
        return fopen($path, 'w');
    }
    
    protected function uploadChunksToDropbox(array $localFiles, string $currentDate)
    {
        $client = new Client(env('DROPBOX_ACCESS_TOKEN'));
        $remoteDir = '/Frames Export/' . $currentDate . '/';
        $uploadedPaths = [];
        
        try {
            $client->createFolder($remoteDir);
        } catch (\Exception $e) {
            // Folder likely exists
        }
        
        foreach ($localFiles as $localFile) {
            if (!file_exists($localFile)) {
                $this->error("File not found: $localFile");
                continue;
            }
            
            $remotePath = $remoteDir . basename($localFile);
            $this->info("Uploading: " . basename($localFile) . " (" . $this->formatBytes(filesize($localFile)) . ")");
            
            try {
                $this->uploadWithRetry($client, $localFile, $remotePath);
                $uploadedPaths[] = $remotePath;
            } catch (\Exception $e) {
                $this->error("Failed to upload " . basename($localFile) . ": " . $e->getMessage());
                throw $e;
            }
        }
        
        return $uploadedPaths;
    }

    protected function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, $precision) . ' ' . $units[$pow];
    }

    protected function uploadWithRetry(Client $client, string $localPath, string $remotePath, int $maxRetries = 3, int $delay = 5)
    {
        $retryCount = 0;
        $fileSize = filesize($localPath);
        
        while ($retryCount < $maxRetries) {
            try {
                $this->info("Uploading: " . basename($localPath) . " (" . $this->formatBytes($fileSize) . ")");
                
                // For files larger than 150MB, use chunked upload
                if ($fileSize > 150 * 1024 * 1024) {
                    $this->info("Using chunked upload for large file");
                    $handle = fopen($localPath, 'rb');
                    $client->uploadChunked($remotePath, $handle, $fileSize);
                    fclose($handle);
                } else {
                    $fileContents = file_get_contents($localPath);
                    $client->upload($remotePath, $fileContents);
                }
                
                $this->info("Successfully uploaded: " . basename($localPath));
                return true;
                
            } catch (\Exception $e) {
                $retryCount++;
                if ($retryCount >= $maxRetries) {
                    throw new \Exception("Upload failed after $maxRetries attempts: " . $e->getMessage());
                }
                
                $this->warn("Upload attempt $retryCount failed for " . basename($localPath) . ". Retrying in {$delay} seconds...");
                sleep($delay);
            }
        }
    }
    
    protected function mergeDropboxCsvs(array $chunkPaths, string $currentDate)
    {
        $this->info("Starting merge process with " . count($chunkPaths) . " chunks");
        $this->info("Memory usage at start: " . memory_get_usage(true) / 1024 / 1024 . "MB");
        $this->info("Memory limit: " . ini_get('memory_limit'));
        
        $client = new Client(env('DROPBOX_ACCESS_TOKEN'));
        $finalPath = '/Frames Export/frames_merged_' . now()->format('Y-m-d_H') . '.zip'; // Changed to .zip
        $remoteDir = '/Frames Export/' . $currentDate . '/';
        set_time_limit(0);
        
        try {
            $startTime = microtime(true);
            
            // Download headers from first file
            $this->info("Downloading first chunk for headers...");
            $firstChunkContent = $client->download($chunkPaths[0]);
            if (is_resource($firstChunkContent)) {
                $firstChunkContent = stream_get_contents($firstChunkContent);
            }
            $this->info("First chunk downloaded in " . round(microtime(true) - $startTime, 2) . "s");

            // Extract headers
            $headers = str_getcsv(explode("\n", $firstChunkContent)[0]);
            if (empty($headers)) {
                throw new \Exception("Could not extract headers from first chunk");
            }

            // Create a temporary file for the merged content
            $mergedTempPath = tempnam(sys_get_temp_dir(), 'merged_csv');
            $this->info("Created temporary merge file at: " . $mergedTempPath);
            
            $mergedFile = fopen($mergedTempPath, 'w');
            fputcsv($mergedFile, $headers);
            fwrite($mergedFile, "\n");
            
            $chunkCount = 0;
            $totalLines = 0;
            
            // Append each chunk's content (without headers)
            foreach ($chunkPaths as $chunkPath) {
                $chunkStart = microtime(true);
                $this->info("Processing chunk: " . basename($chunkPath));
                
                $chunkStream = $client->download($chunkPath);
                if (is_resource($chunkStream)) {
                    $firstLine = true;
                    while (!feof($chunkStream)) {
                        $line = fgets($chunkStream);
                        if ($firstLine) {
                            $firstLine = false;
                            continue; // Skip header
                        }
                        if (!empty(trim($line))) {
                            fwrite($mergedFile, $line);
                            $totalLines++;
                        }
                    }
                    fclose($chunkStream);
                }
                
                $this->info("Processed chunk in " . round(microtime(true) - $chunkStart, 2) . "s");
            }
            
            fclose($mergedFile);
            $this->info("Merged all chunks. Total lines: {$totalLines}");
            
            // Create ZIP file
            $zipStart = microtime(true);
            $zipPath = $mergedTempPath . '.zip';
            $this->info("Creating ZIP archive at: {$zipPath}");
            
            $zip = new ZipArchive();
            if ($zip->open($zipPath, ZipArchive::CREATE) !== TRUE) {
                throw new \Exception("Cannot create ZIP file at {$zipPath}");
            }
            
            $zip->addFile($mergedTempPath, 'frames_export_' . now()->format('Y-m-d_H-i-s') . '.csv');
            $zip->close();
            $this->info("ZIP created in " . round(microtime(true) - $zipStart, 2) . "s");
            
            // Get file sizes for comparison
            $originalSize = filesize($mergedTempPath);
            $compressedSize = filesize($zipPath);
            $compressionRatio = round(($originalSize - $compressedSize) / $originalSize * 100, 2);
            
            $this->info(sprintf(
                "Compression results: Original: %s | Compressed: %s | Ratio: %s%%",
                $this->formatBytes($originalSize),
                $this->formatBytes($compressedSize),
                $compressionRatio
            ));
            
            // Upload the ZIP file
            $uploadStart = microtime(true);
            $this->info("Starting final upload of ZIP file...");
            $this->info("Final file size: " . $this->formatBytes(filesize($zipPath)));
            $this->info("Memory usage before upload: " . round(memory_get_usage(true) / 1024 / 1024, 2) . "MB");
            
            $this->uploadWithRetry($client, $zipPath, $finalPath);
            
            $this->info("Upload completed in " . round(microtime(true) - $uploadStart, 2) . "s");
            
            // Clean up chunks and date folder
            $this->cleanDropboxChunks($client, $chunkPaths);
            $this->deleteDropboxFolder($client, $remoteDir);
            
            // Delete temporary files
            unlink($mergedTempPath);
            unlink($zipPath);
            
            $totalTime = round(microtime(true) - $startTime, 2);
            $this->info("Merge process completed in {$totalTime}s. Final file: {$finalPath}");
            
            return $finalPath;
            
        } catch (\Exception $e) {
            // Clean up any temporary files if they exist
            if (isset($mergedTempPath) && file_exists($mergedTempPath)) {
                @unlink($mergedTempPath);
            }
            if (isset($zipPath) && file_exists($zipPath)) {
                @unlink($zipPath);
            }
            throw new \Exception("Merge failed: " . $e->getMessage());
        }
    }

    protected function cleanDropboxChunks(Client $client, array $chunkPaths)
    {
        foreach ($chunkPaths as $chunkPath) {
            try {
                $client->delete($chunkPath);
                $this->info("Deleted Dropbox chunk: " . basename($chunkPath));
            } catch (\Exception $e) {
                $this->warn("Could not delete Dropbox chunk: " . basename($chunkPath));
            }
        }
    }

    protected function deleteDropboxFolder(Client $client, string $folderPath)
    {
        try {
            $client->delete($folderPath);
            $this->info("Deleted Dropbox folder: " . $folderPath);
        } catch (\Exception $e) {
            $this->warn("Could not delete Dropbox folder: " . $folderPath);
        }
    }

    protected function getCsvHeaders()
    {
        // Define our base set of headers
        $staticHeaders = [
            'Package', 'Tier', 'Price', 'Charge', 'Manufacturer', 'Brand', 'Model',
            'Frame Color', 'Size', 'Frame Code', 'A', 'B', 'ED', 'DBL', 'EDAngle',
            'RIM', 'Edge Type', 'Image', 'Configuration FPC', 'UPC', 'SKU', 'Status',
            'Material', 'Precious Metal', 'Frame Shape', 'RX', 'Frame Type',
            'Lens Color', 'Lens Color Code', 'FrameColorGroup', 'FrameColor',
            'EyeSize', 'TempleLength', 'BridgeSize'
        ];

        // Get additional dynamic headers from the JSON data
        $dynamicHeaders = FrameStyles::query()
            ->take(1000)
            ->get()
            ->flatMap(function ($style) {
                $data = json_decode($style->frame_style_data, true) ?: [];
                return array_keys($data);
            })
            ->unique()
            ->filter(function ($header) use ($staticHeaders) {
                return !in_array($header, $staticHeaders);
            })
            ->values()
            ->toArray();

        return array_merge($staticHeaders, $dynamicHeaders);
    }


    protected function formatStyleRow($style, $headers)
    {
        $jsonData = json_decode($style->frame_style_data, true) ?: [];
        $packageData = json_decode($style->package_data, true);
        
        // Base fields with fallbacks
        $baseFields = [
            'Package' => is_array($packageData) ? implode(' | ', $packageData) : ($packageData ?? '-'),
            'Tier' => $style->tier ?? '-',
            'Price' => $style->price ?? '-',
            'Charge' => $style->charge ?? '-',
            'Model' => $style->name ?? '-',
            'Frame Color' => $style->color ?? '-',
            'Image' => $style->image ?? '-',
            'Configuration FPC' => $style->configuration_fpc ?? '-',
            'Status' => $style->status ?? '-',
            'Size' => isset($jsonData['EyeSize'], $jsonData['BridgeSize'], $jsonData['TempleLength'])
                ? $jsonData['EyeSize'].'-'.$jsonData['BridgeSize'].'-'.$jsonData['TempleLength']
                : '-',
        ];

        // Common JSON fields with standardized keys
        $jsonFields = [
            'Manufacturer' => $jsonData['ManufacturerName'] ?? '-',
            'Brand' => $jsonData['BrandName'] ?? '-',
            'Frame Code' => $jsonData['FrameColorCode'] ?? '-',
            'A' => $jsonData['A'] ?? '-',
            'B' => $jsonData['B'] ?? '-',
            'ED' => $jsonData['ED'] ?? '-',
            'DBL' => $jsonData['DBL'] ?? '-',
            'EDAngle' => $jsonData['EDAngle'] ?? '-',
            'RIM' => $jsonData['Rim'] ?? '-',
            'Edge Type' => $jsonData['EdgeType'] ?? '-',
            'UPC' => $jsonData['UPC'] ?? '-',
            'SKU' => $jsonData['SKU'] ?? '-',
            'Material' => $jsonData['Material'] ?? '-',
            'Precious Metal' => $jsonData['PreciousMetal'] ?? '-',
            'Frame Shape' => $jsonData['FrameShape'] ?? '-',
            'RX' => $jsonData['RX'] ?? '-',
            'Frame Type' => $jsonData['FrameType'] ?? '-',
            'Lens Color' => $jsonData['LensColor'] ?? '-',
            'Lens Color Code' => $jsonData['LensColorCode'] ?? '-',
            'FrameColorGroup' => $jsonData['FrameColorGroup'] ?? '-',
            'FrameColor' => $jsonData['FrameColor'] ?? '-',
            'EyeSize' => $jsonData['EyeSize'] ?? '-',
            'TempleLength' => $jsonData['TempleLength'] ?? '-',
            'BridgeSize' => $jsonData['BridgeSize'] ?? '-',
        ];

        // Combine all fields
        $allFields = array_merge($baseFields, $jsonFields);
        
        // Add any remaining dynamic fields from JSON
        foreach ($jsonData as $key => $value) {
            if (!array_key_exists($key, $allFields)) {
                $allFields[$key] = is_array($value) ? json_encode($value) : $value;
            }
        }

        // Build the final row in correct header order
        $row = [];
        foreach ($headers as $header) {
            $row[] = $allFields[$header] ?? '-';
        }

        return $row;
    }
}