<?php 
// app/Custom/Excel/LsWebOrders.php
namespace App\Custom\Excel;

use Illuminate\Database\Eloquent\Collection;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithMapping;
use App\Models\Backend\Order;
use App\Models\Backend\StockOrder;
use App\Models\Backend\RxOrders;
use App\Models\Backend\ConsumablesOrder;

class LsWebOrders implements FromCollection, WithHeadings, ShouldAutoSize, WithMapping
{
    use Exportable;

    protected $maxAdditionalItems = 0;
    protected $maxRepairsServices = 0;
    protected $headings = null;

    /**
     * Fetch the collection for the data export.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function collection(): Collection
    {
        return Order::where('online', 1)
            ->where('is_saved', 0)
            ->orderBy('created_at', 'desc')
            ->take(5000)
            ->get();
    }

    /**
     * Calculate the maximum number of additional items and repairs services.
     */
    protected function calculateMaxItems(): void
    {
        if ($this->maxAdditionalItems > 0 || $this->maxRepairsServices > 0) {
            return;
        }

        $orders = $this->collection();
        foreach ($orders as $order) {
            if ($order->type == Order::ORDER_RX) {
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                if ($rxOrder) {
                    $rxOrderArray = json_decode($rxOrder->data, true) ?? [];
                    if (isset($rxOrderArray['prices']['additional'])) {
                        $this->maxAdditionalItems = max($this->maxAdditionalItems, count($rxOrderArray['prices']['additional']));
                    }
                    if (isset($rxOrderArray['prices']['repairs_services'])) {
                        $this->maxRepairsServices = max($this->maxRepairsServices, count($rxOrderArray['prices']['repairs_services']));
                    }
                }
            }
        }
    }

    /**
     * Define the column headings for the CSV file.
     *
     * @return array
     */
    public function headings(): array
    {
        if ($this->headings !== null) {
            return $this->headings; // Return cached headings
        }

        $baseHeadings = [
            // Common
            'Order_Number', 'Acc_Number', 'Patient', 'Order_Date_Time', 'order_type',
            // Consumables
            'consumables_name', 'consumables_sub_name', 'consumables_description',
            'consumables_quantity', 'consumables_image', 'consumables_image_hover', 'consumables_upc',
            // Stock
            'stock_l_material', 'stock_l_colour', 'stock_l_coating', 'stock_l_sph', 'stock_l_cyl', 'stock_l_size',
            'stock_l_quantity', 'stock_l_price', 'stock_l_barcode', 'stock_l_desc', 'stock_l_vendor', 'stock_l_cards',
            'stock_r_material', 'stock_r_colour', 'stock_r_coating', 'stock_r_sph', 'stock_r_cyl', 'stock_r_size',
            'stock_r_quantity', 'stock_r_price', 'stock_r_barcode', 'stock_r_desc', 'stock_r_vendor', 'stock_r_cards',
            // Common
            'price_grand_total',
            // RX
            'rx_first_name', 'rx_last_name', 'rx_package', 'rx_multipair_ref', 'rx_distance_right_sph_new', 'rx_distance_left_sph_new',
            'rx_distance_right_dist_type', 'rx_distance_right_sph', 'rx_distance_right_cyl',
            'rx_distance_right_axis', 'rx_distance_right_strength', 'rx_distance_left_dist_type',
            'rx_distance_left_sph', 'rx_distance_left_cyl', 'rx_distance_left_axis',
            'rx_add_for_reading_right_add', 'rx_add_for_reading_left_add', 'rx_add_for_reading_right_boc',
            'rx_add_for_reading_left_boc', 'rx_extended_info_right_pd', 'rx_extended_info_right_intermediate_pd',
            'rx_extended_info_right_npd', 'rx_extended_info_right_oc', 'rx_extended_info_left_pd',
            'rx_extended_info_left_intermediate_pd', 'rx_extended_info_left_npd', 'rx_extended_info_left_oc',
            'rx_extended_info_right_base_curve', 'rx_prism_right_prism1', 'rx_prism_right_base1',
            'rx_prism_right_prism2', 'rx_prism_right_base2', 'rx_prism_left_prism1', 'rx_prism_left_base1',
            'rx_prism_left_prism2', 'rx_prism_left_base2', 'rx_final_rx_right_sph', 'rx_final_rx_right_cyl',
            'rx_final_rx_right_axis', 'rx_final_rx_left_sph', 'rx_final_rx_left_cyl', 'rx_final_rx_left_axis',
            'rx_lens_lens_type', 'rx_lens_sub_type', 'rx_lens_lens_name', 'rx_lens_lens_material', 'rx_lens_lens_filter',
            'rx_lens_coating_name', 'rx_frame_job_type', 'rx_frame_invoice_order_no', 'rx_frame_frame_info_brand',
            'rx_frame_frame_info_model', 'rx_frame_frame_info_colour', 'rx_frame_frame_type', 'rx_frame_ft_other',
            'rx_frame_frame_size_a', 'rx_frame_frame_size_b', 'rx_frame_frame_size_ed', 'rx_frame_frame_size_dbl',
            'rx_frame_frame_size_shape_factor', 'rx_frame_specify_thickness_r_ct_or_et', 'rx_frame_specify_thickness_l_ct_or_et',
            'rx_frame_bevel_type', 'rx_additional_items_clip_on_type', 'rx_additional_items_clip_on_lens_type',
            'rx_additional_items_clip_on_lens_colour', 'rx_additional_items_repair_description', 'rx_special_instructions',
            'price_rx_lenses_total', 'price_rx_right_lens', 'price_rx_left_lens', 'prices_custom_clip_on',
        ];

        // Calculate max items for dynamic headings
        $this->calculateMaxItems();

        // Add dynamic headings for additional items
        $additionalHeadings = [];
        for ($i = 1; $i <= $this->maxAdditionalItems; $i++) {
            $additionalHeadings[] = "additionalProduct{$i}";
            $additionalHeadings[] = "additionalProduct{$i}Price";
        }

        // Add dynamic headings for repairs services
        $repairsHeadings = [];
        for ($i = 1; $i <= $this->maxRepairsServices; $i++) {
            $repairsHeadings[] = "repairsService{$i}";
            $repairsHeadings[] = "repairsService{$i}Price";
        }

        $this->headings = array_merge($baseHeadings, $additionalHeadings, $repairsHeadings);
        return $this->headings;
    }

    /**
     * Map the row data for the Excel file.
     *
     * @param $row
     * @return array
     */
    public function map($row): array
    {
        $array = [
            'Order_Number' => $row->getOrderNumber(),
            'Acc_Number' => $row->user->ac_code,
            'Patient' => $row->patient,
            'Order_Date_Time' => date_time_format($row->created_at, 'd/m/Y H:i'),
        ];
        $emptyColumns = array_fill_keys($this->headings(), '');

        if ($row->type == Order::ORDER_CONSUMABLE) {
            $consumableOrders = ConsumablesOrder::where('order_id', $row->id)->get();
    
            return $consumableOrders->map(function ($consumableOrder) use ($array, $emptyColumns) {
                return array_merge($emptyColumns, $array, [
                    'order_type' => Order::ORDER_CONSUMABLE,
                    'consumables_name' => $consumableOrder->name ?? '',
                    'consumables_sub_name' => $consumableOrder->sub_name ?? '',
                    'consumables_description' => $consumableOrder->description ?? '',
                    'consumables_quantity' => $consumableOrder->quantity ?? '',
                    'consumables_image' => $consumableOrder->image ?? '',
                    'consumables_image_hover' => $consumableOrder->image_hover ?? '',
                    'price_grand_total' => $consumableOrder->price ?? '',
                    'consumables_upc' => $consumableOrder->upc ?? '',
                ]);
            })->toArray();
        }

        if ($row->type == Order::ORDER_STOCK) {
            $stockOrder = StockOrder::where('order_id', $row->id)->first();
            // Calculate price_grand_total based on eye type
            $priceGrandTotal = 0;
            if ($stockOrder->type === 'r_price') {
                $priceGrandTotal = $stockOrder->r_price ?? 0;
            } elseif ($stockOrder->type === 'l_size') {
                $priceGrandTotal = $stockOrder->l_price ?? 0;
            } else {
                // For both eyes or unspecified type, sum both prices
                $priceGrandTotal = ($stockOrder->l_price ?? 0) + ($stockOrder->r_price ?? 0);
            }
            return array_merge($emptyColumns, $array, [
                'order_type' => Order::ORDER_STOCK,
                'stock_l_material' => $stockOrder->l_material ?? '',
                'stock_l_colour' => $stockOrder->l_colour ?? '',
                'stock_l_coating' => $stockOrder->l_coating ?? '',
                'stock_l_sph' => $stockOrder->l_sph ?? '',
                'stock_l_cyl' => $stockOrder->l_cyl ?? '',
                'stock_l_size' => $stockOrder->l_size ?? '',
                'stock_l_quantity' => $stockOrder->l_quantity ?? '',
                'stock_l_price' => $stockOrder->l_price ?? '',
                'stock_l_barcode' => $stockOrder->l_barcode ?? '',
                'stock_l_desc' => $stockOrder->l_desc ?? '',
                'stock_l_vendor' => $stockOrder->l_vendor ?? '',
                'stock_l_cards' => $stockOrder->l_cards ?? '',
                'stock_r_material' => $stockOrder->r_material ?? '',
                'stock_r_colour' => $stockOrder->r_colour ?? '',
                'stock_r_coating' => $stockOrder->r_coating ?? '',
                'stock_r_sph' => $stockOrder->r_sph ?? '',
                'stock_r_cyl' => $stockOrder->r_cyl ?? '',
                'stock_r_size' => $stockOrder->r_size ?? '',
                'stock_r_quantity' => $stockOrder->r_quantity ?? '',
                'stock_r_price' => $stockOrder->r_price ?? '',
                'stock_r_barcode' => $stockOrder->r_barcode ?? '',
                'stock_r_desc' => $stockOrder->r_desc ?? '',
                'stock_r_vendor' => $stockOrder->r_vendor ?? '',
                'stock_r_cards' => $stockOrder->r_cards ?? '',
                'price_grand_total' => $priceGrandTotal,
            ]);
        }

        if ($row->type == Order::ORDER_RX) {
            $rxOrder = RxOrders::where('order_id', $row->id)->first();
            $rxOrderArray = json_decode($rxOrder->data, true) ?? [];
            $rxOrderArray = $this->replaceNaNWithEmpty($rxOrderArray);

            // Map additional items dynamically
            $additionalItems = [];
            if (isset($rxOrderArray['prices']['additional'])) {
                $i = 1;
                foreach ($rxOrderArray['prices']['additional'] as $item) {
                    $additionalItems["additionalProduct{$i}"] = $item['label'] ?? '';
                    $additionalItems["additionalProduct{$i}Price"] = $item['price'] ?? '';
                    $i++;
                }
            }

            // Map repairs services dynamically
            $repairsItems = [];
            if (isset($rxOrderArray['prices']['repairs_services'])) {
                $i = 1;
                foreach ($rxOrderArray['prices']['repairs_services'] as $item) {
                    $repairsItems["repairsService{$i}"] = $item['label'] ?? '';
                    $repairsItems["repairsService{$i}Price"] = $item['price'] ?? '';
                    $i++;
                }
            }

            return array_merge($emptyColumns, $array, [
                'order_type' => Order::ORDER_RX,
                'price_grand_total' => $rxOrderArray['price'] ?? '',
                'price_rx_lenses_total' => $rxOrderArray['prices']['total_price'] ?? '',
                'price_rx_right_lens' => $rxOrderArray['prices']['right_lens_type'] ?? '',
                'price_rx_left_lens' => $rxOrderArray['prices']['left_lens_type'] ?? '',
                'prices_custom_clip_on' => $rxOrderArray['prices']['custom_clip_on'] ?? '',
                'rx_first_name' => $rxOrderArray['first_name'] ?? '',
                'rx_last_name' => $rxOrderArray['last_name'] ?? '',
                'rx_package' => $rxOrderArray['package'] ?? '',
                'rx_multipair_ref' => $rxOrderArray['multipair_ref'] ?? '',
                'rx_distance_right_sph_new' => $rxOrderArray['rx']['distance']['right_sph_new'] ?? '',
                'rx_distance_left_sph_new' => $rxOrderArray['rx']['distance']['left_sph_new'] ?? '',
                'rx_distance_right_dist_type' => $rxOrderArray['rx']['distance']['right_dist_type'] ?? '',
                'rx_distance_right_sph' => $rxOrderArray['rx']['distance']['right_sph'] ?? '',
                'rx_distance_right_cyl' => $rxOrderArray['rx']['distance']['right_cyl'] ?? '',
                'rx_distance_right_axis' => $rxOrderArray['rx']['distance']['right_axis'] ?? '',
                'rx_distance_right_strength' => $rxOrderArray['rx']['distance']['right_strength'] ?? '',
                'rx_distance_left_dist_type' => $rxOrderArray['rx']['distance']['left_dist_type'] ?? '',
                'rx_distance_left_sph' => $rxOrderArray['rx']['distance']['left_sph'] ?? '',
                'rx_distance_left_cyl' => $rxOrderArray['rx']['distance']['left_cyl'] ?? '',
                'rx_distance_left_axis' => $rxOrderArray['rx']['distance']['left_axis'] ?? '',
                'rx_add_for_reading_right_add' => $rxOrderArray['rx']['add_for_reading']['right_add'] ?? '',
                'rx_add_for_reading_left_add' => $rxOrderArray['rx']['add_for_reading']['left_add'] ?? '',
                'rx_add_for_reading_right_boc' => $rxOrderArray['rx']['add_for_reading']['right_boc'] ?? '',
                'rx_add_for_reading_left_boc' => $rxOrderArray['rx']['add_for_reading']['left_boc'] ?? '',
                'rx_extended_info_right_pd' => $rxOrderArray['rx']['extended_info']['right_pd'] ?? '',
                'rx_extended_info_right_intermediate_pd' => $rxOrderArray['rx']['extended_info']['right_intermediate_pd'] ?? '',
                'rx_extended_info_right_npd' => $rxOrderArray['rx']['extended_info']['right_npd'] ?? '',
                'rx_extended_info_right_oc' => $rxOrderArray['rx']['extended_info']['right_oc'] ?? '',
                'rx_extended_info_left_pd' => $rxOrderArray['rx']['extended_info']['left_pd'] ?? '',
                'rx_extended_info_left_intermediate_pd' => $rxOrderArray['rx']['extended_info']['left_intermediate_pd'] ?? '',
                'rx_extended_info_left_npd' => $rxOrderArray['rx']['extended_info']['left_npd'] ?? '',
                'rx_extended_info_left_oc' => $rxOrderArray['rx']['extended_info']['left_oc'] ?? '',
                'rx_extended_info_right_base_curve' => $rxOrderArray['rx']['extended_info']['right_base_curve'] ?? '',
                'rx_prism_right_prism1' => $rxOrderArray['rx']['prism']['right_prism1'] ?? '',
                'rx_prism_right_base1' => $rxOrderArray['rx']['prism']['right_base1'] ?? '',
                'rx_prism_right_prism2' => $rxOrderArray['rx']['prism']['right_prism2'] ?? '',
                'rx_prism_right_base2' => $rxOrderArray['rx']['prism']['right_base2'] ?? '',
                'rx_prism_left_prism1' => $rxOrderArray['rx']['prism']['left_prism1'] ?? '',
                'rx_prism_left_base1' => $rxOrderArray['rx']['prism']['left_base1'] ?? '',
                'rx_prism_left_prism2' => $rxOrderArray['rx']['prism']['left_prism2'] ?? '',
                'rx_prism_left_base2' => $rxOrderArray['rx']['prism']['left_base2'] ?? '',
                'rx_final_rx_right_sph' => $rxOrderArray['rx']['final_rx']['right_sph'] ?? '',
                'rx_final_rx_right_cyl' => $rxOrderArray['rx']['final_rx']['right_cyl'] ?? '',
                'rx_final_rx_right_axis' => $rxOrderArray['rx']['final_rx']['right_axis'] ?? '',
                'rx_final_rx_left_sph' => $rxOrderArray['rx']['final_rx']['left_sph'] ?? '',
                'rx_final_rx_left_cyl' => $rxOrderArray['rx']['final_rx']['left_cyl'] ?? '',
                'rx_final_rx_left_axis' => $rxOrderArray['rx']['final_rx']['left_axis'] ?? '',
                'rx_lens_lens_type' => $rxOrderArray['lens']['lens_type'] ?? '',
                'rx_lens_sub_type' => $rxOrderArray['lens']['sub_type'] ?? '',
                'rx_lens_lens_name' => $rxOrderArray['lens']['lens_name'] ?? '',
                'rx_lens_lens_material' => $rxOrderArray['lens']['lens_material'] ?? '',
                'rx_lens_lens_filter' => $rxOrderArray['lens']['lens_filter'] ?? '',
                'rx_lens_coating_name' => $rxOrderArray['lens']['coating_name'] ?? '',
                'rx_frame_job_type' => $rxOrderArray['frame']['job_type'] ?? '',
                'rx_frame_invoice_order_no' => $rxOrderArray['frame']['invoice_order_no'] ?? '',
                'rx_frame_frame_info_brand' => $rxOrderArray['frame']['frame_info']['brand'] ?? '',
                'rx_frame_frame_info_model' => $rxOrderArray['frame']['frame_info']['model'] ?? '',
                'rx_frame_frame_info_colour' => $rxOrderArray['frame']['frame_info']['colour'] ?? '',
                'rx_frame_frame_type' => $rxOrderArray['frame']['frame_type'] ?? '',
                'rx_frame_ft_other' => $rxOrderArray['frame']['ft_other'] ?? '',
                'rx_frame_frame_size_a' => $rxOrderArray['frame']['frame_size']['a'] ?? '',
                'rx_frame_frame_size_b' => $rxOrderArray['frame']['frame_size']['b'] ?? '',
                'rx_frame_frame_size_ed' => $rxOrderArray['frame']['frame_size']['ed'] ?? '',
                'rx_frame_frame_size_dbl' => $rxOrderArray['frame']['frame_size']['dbl'] ?? '',
                'rx_frame_frame_size_shape_factor' => $rxOrderArray['frame']['frame_size']['shape_factor'] ?? '',
                'rx_frame_specify_thickness_r_ct_or_et' => $rxOrderArray['frame']['specify_thickness']['r_ct_or_et'] ?? '',
                'rx_frame_specify_thickness_l_ct_or_et' => $rxOrderArray['frame']['specify_thickness']['l_ct_or_et'] ?? '',
                'rx_frame_bevel_type' => $rxOrderArray['frame']['bevel_type'] ?? '',
                'rx_additional_items_clip_on_type' => $rxOrderArray['additional_items']['clip_on_type'] ?? '',
                'rx_additional_items_clip_on_lens_type' => $rxOrderArray['additional_items']['clip_on_lens_type'] ?? '',
                'rx_additional_items_clip_on_lens_colour' => $rxOrderArray['additional_items']['clip_on_lens_colour'] ?? '',
                'rx_additional_items_repair_description' => $rxOrderArray['additional_items']['repair_description'] ?? '',
                'rx_special_instructions' => $rxOrderArray['special_instructions'] ?? '',
            ], $additionalItems, $repairsItems);
        }
    }

    public function replaceNaNWithEmpty($data) {
        foreach ($data as $key => $value) {
            if (is_array($value) || is_object($value)) {
                $data[$key] = self::replaceNaNWithEmpty((array) $value);
            } elseif ($value === null || (is_string($value) && ($value === '[object Object]' || $value === 'object Object' || $value === 'NaN'))) {
                $data[$key] = '';
            }
        }
        return $data;
    }
}