<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Backend\Order;
use App\Models\Backend\ConsumablesOrder;
use App\Models\Backend\ImportedConsumablesOrder;
use App\Models\Backend\OrderStatus;
use App\Models\Backend\StockOrder;
use App\Models\Backend\RxOrders;
use App\Exceptions\GeneralException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Models\Auth\User;
use App\Models\Backend\Stockvalidation;
use App\Models\Backend\RxValidations;

class OrderController extends Controller
{
    /**
     * Create a new consumable order
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createConsumableOrder(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'orderData' => 'required|array|min:1',
                'orderData.*.itemId' => 'required|exists:imported_consumables_orders,upc',
                'orderData.*.itemQty' => 'required|integer|min:1|max:20',
                'order_reference' => 'nullable|string|max:255',
                'notes' => 'nullable|string',
                'account_number' => 'required|exists:users,ac_code',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $user = User::where('ac_code', $request->account_number)->first();

            // Get the API client from the request (set by middleware)
            $apiClient = $request->get('api_client');
            if (!$apiClient) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid API client'
                ], 400);
            }

            DB::beginTransaction();

            $totalPrice = 0;
            $orderItems = [];

            // Validate and prepare order items
            foreach ($request->orderData as $item) {
                $product = ImportedConsumablesOrder::where('upc', $item['itemId'])->first();
                if (!$product) {
                    throw new GeneralException("Product with UPC {$item['itemId']} not found");
                }

                // Validate quantity limits from the product
                if ($item['itemQty'] < 1 || $item['itemQty'] > 20) {
                    throw new GeneralException("Quantity for product {$product->name} must be between {$product->min} and {$product->max}");
                }

                $itemPrice = $product->price * $item['itemQty'];
                $totalPrice += $itemPrice;

                $orderItems[] = [
                    'product' => $product,
                    'quantity' => $item['itemQty'],
                    'price' => $product->price,
                ];
            }

            // Create the main order - using a default user or system user
            $order = Order::create([
                'user_id' => $user->id,
                'type' => Order::ORDER_CONSUMABLE,
                'method' => 'W',
                'status' => OrderStatus::getStatus(OrderStatus::STATUS_SUBMITTED)->id,
                'price' => $totalPrice,
                'patient' => $request->order_reference,
                'multiple' => count($orderItems) > 1
            ]);

            // Create consumable order items
            foreach ($orderItems as $item) {
                ConsumablesOrder::create([
                    'order_id' => $order->id,
                    'name' => $item['product']->name,
                    'sub_name' => $item['product']->sub_name,
                    'description' => $item['product']->description,
                    'image' => $item['product']->image,
                    'image_hover' => $item['product']->image_hover,
                    'upc' => $item['product']->upc,
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Consumable order created successfully',
                'data' => [
                    'order_id' => $order->getOrderNumber(),
                    'total_price' => $totalPrice,
                    'status' => $order->statusModel->name ?? 'Submitted',
                    'created_at' => $order->created_at,
                    'items_count' => count($orderItems),
                    'order_reference' => $request->order_reference,
                    'items' => collect($orderItems)->map(function($item) {
                        return [
                            'product_name' => $item['product']->name,
                            'variant_name' => $item['product']->sub_name,
                            'upc' => $item['product']->upc,
                            'quantity' => $item['quantity'],
                            'unit_price' => $item['price'],
                            'total_price' => $item['price'] * $item['quantity']
                        ];
                    })
                ]
            ], 201);

        } catch (GeneralException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while creating the order',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Create a new stock order
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createStockOrder(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'orderData' => 'required|array|min:1',
                'orderData.*.eye' => 'required|in:left,right',
                'orderData.*.upc' => 'required|string',
                'orderData.*.quantity' => 'required|integer|min:1',
                'order_reference' => 'nullable|string|max:255',
                'account_number' => 'required|exists:users,ac_code',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Get the API client from the request (set by middleware)
            $apiClient = $request->get('api_client');
            if (!$apiClient) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid API client'
                ], 400);
            }

            $user = User::where('ac_code', $request->account_number)->first();

            DB::beginTransaction();

            $totalPrice = 0;
            $stockItemsData = [];
            $leftEye = null;
            $rightEye = null;

            foreach ($request->orderData as $item) {
                $stockValidation = Stockvalidation::where('barcode', $item['upc'])->first();

                if (!$stockValidation) {
                    throw new GeneralException("Stock item with UPC {$item['upc']} not found");
                }

                $itemPrice = ($stockValidation->price / 2) * $item['quantity'];
                $totalPrice += $itemPrice;

                $eyeData = [
                    'material' => $stockValidation->material,
                    'colour' => $stockValidation->colour,
                    'coating' => $stockValidation->coating,
                    'sph' => $stockValidation->sph,
                    'cyl' => $stockValidation->cyl,
                    'quantity' => $item['quantity'],
                    'size' => $stockValidation->size,
                    'price' => $itemPrice,
                    'barcode' => $stockValidation->barcode,
                    'vendor' => $stockValidation->vendor,
                    'cards' => $stockValidation->cards,
                ];

                if ($item['eye'] === 'left') {
                    $leftEye = $eyeData;
                } else {
                    $rightEye = $eyeData;
                }
            }
            
            $orderType = StockOrder::BOTH_EYES;
            if(!$leftEye){
                $orderType = StockOrder::RIGHT_EYE;
            } else if(!$rightEye) {
                $orderType = StockOrder::LEFT_EYE;
            }

            $order = Order::create([
                'user_id' => $user->id,
                'type' => Order::ORDER_STOCK,
                'method' => 'W',
                'status' => OrderStatus::getStatus(OrderStatus::STATUS_SUBMITTED)->id,
                'price' => $totalPrice,
                'patient' => $request->order_reference,
                'multiple' => $leftEye && $rightEye,
            ]);

            StockOrder::create([
                'order_id' => $order->id,
                'type' => $orderType,
                'l_material'    => $leftEye['material'] ?? null,
                'l_colour'      => $leftEye['colour'] ?? null,
                'l_coating'     => $leftEye['coating'] ?? null,
                'l_sph'         => $leftEye['sph'] ?? null,
                'l_cyl'         => $leftEye['cyl'] ?? null,
                'l_quantity'    => $leftEye['quantity'] ?? null,
                'l_size'        => $leftEye['size'] ?? null,
                'l_price'       => $leftEye['price'] ?? null,
                'l_barcode'     => $leftEye['barcode'] ?? null,
                'l_vendor'      => $leftEye['vendor'] ?? null,
                'l_cards'       => $leftEye['cards'] ?? null,
                'r_material'    => $rightEye['material'] ?? null,
                'r_colour'      => $rightEye['colour'] ?? null,
                'r_coating'     => $rightEye['coating'] ?? null,
                'r_sph'         => $rightEye['sph'] ?? null,
                'r_cyl'         => $rightEye['cyl'] ?? null,
                'r_quantity'    => $rightEye['quantity'] ?? null,
                'r_size'        => $rightEye['size'] ?? null,
                'r_price'       => $rightEye['price'] ?? null,
                'r_barcode'     => $rightEye['barcode'] ?? null,
                'r_vendor'      => $rightEye['vendor'] ?? null,
                'r_cards'       => $rightEye['cards'] ?? null,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Stock order created successfully',
                'data' => [
                    'order_id' => $order->getOrderNumber(),
                    'total_price' => $totalPrice,
                    'status' => $order->statusModel->name ?? 'Submitted',
                    'created_at' => $order->created_at,
                    'items_count' => count($request->orderData),
                    'order_reference' => $request->order_reference,
                    'items' => collect($request->orderData)->map(function($item) {
                        $stockValidation = Stockvalidation::where('barcode', $item['upc'])->first();
                        return [
                            'eye' => $item['eye'],
                            'upc' => $item['upc'],
                            'quantity' => $item['quantity'],
                            'material' => $stockValidation->material,
                            'colour' => $stockValidation->colour,
                            'coating' => $stockValidation->coating,
                            'sph' => $stockValidation->sph,
                            'cyl' => $stockValidation->cyl,
                            'unit_price' => $stockValidation->price / 2,
                            'total_price' => ($stockValidation->price / 2) * $item['quantity']
                        ];
                    })
                ]
            ], 200);

        } catch (GeneralException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while creating the order',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Create a new Rx order
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createRxOrder(Request $request)
    {
        // 1. Validation
        $validator = Validator::make($request->all(), [
            // Patient Info
            'first_name' => 'required|string|max:191',
            'last_name' => 'required|string|max:191',
            'account_number' => 'required|exists:users,ac_code',

            // Rx (distance, add_for_reading, extended_info, prism, final_rx)
            'rx.distance.right_sph' => 'required|numeric',
            'rx.distance.left_sph' => 'required|numeric',
            'rx.distance.right_cyl' => 'required|numeric',
            'rx.distance.left_cyl' => 'required|numeric',
            'rx.distance.right_axis' => 'required|integer|min:0|max:180',
            'rx.distance.left_axis' => 'required|integer|min:0|max:180',

            'rx.distance.right_dist_type' => 'nullable|string|max:50',
            'rx.distance.left_dist_type' => 'nullable|string|max:50',
            'rx.distance.right_strength' => 'nullable|numeric',
            // ...add all other rx fields as needed

            'rx.add_for_reading.right_add' => 'nullable|numeric',
            'rx.add_for_reading.left_add' => 'nullable|numeric',
            'rx.add_for_reading.right_intermediate_strength' => 'nullable|numeric',
            'rx.add_for_reading.left_intermediate_strength' => 'nullable|numeric',
            'rx.add_for_reading.right_boc' => 'nullable|numeric',
            'rx.add_for_reading.left_boc' => 'nullable|numeric',

            'rx.extended_info.right_pd' => 'nullable|numeric',
            'rx.extended_info.left_pd' => 'nullable|numeric',
            'rx.extended_info.right_intermediate_pd' => 'nullable|numeric',
            'rx.extended_info.left_intermediate_pd' => 'nullable|numeric',
            'rx.extended_info.right_oc' => 'nullable|numeric',
            'rx.extended_info.left_oc' => 'nullable|numeric',
            'rx.extended_info.right_base_curve' => 'nullable|string|max:50',

            'rx.prism.right_prism1' => 'nullable|numeric',
            'rx.prism.right_base1' => 'nullable|string|max:10',
            'rx.prism.right_prism2' => 'nullable|numeric',
            'rx.prism.right_base2' => 'nullable|string|max:10',
            'rx.prism.left_prism1' => 'nullable|numeric',
            'rx.prism.left_base1' => 'nullable|string|max:10',
            'rx.prism.left_prism2' => 'nullable|numeric',
            'rx.prism.left_base2' => 'nullable|string|max:10',

            // Lens Codes (all required)
            'material_code' => 'required|string',
            'lens_code'     => 'required|string',
            'colour_code'   => 'required|string',
            'vendor_code'   => 'required|string',
            'coating_code'  => 'required|string',

            // Frame Info
            'frame.job_type' => 'required|string|max:191',
            'frame.invoice_order_no' => 'nullable|string|max:191',
            'frame.frame_info.brand' => 'nullable|string|max:191',
            'frame.frame_info.model' => 'nullable|string|max:191',
            'frame.frame_info.colour' => 'nullable|string|max:191',
            'frame.frame_info.size' => 'nullable|string|max:191',
            'frame.frame_type' => 'required|string|max:191',
            // 'frame.ft_other' => 'nullable|string|max:191',
            'frame.frame_size.a' => 'nullable|numeric',
            'frame.frame_size.b' => 'nullable|numeric',
            'frame.frame_size.ed' => 'nullable|numeric',
            'frame.frame_size.dbl' => 'nullable|numeric',


            // Add-ons
            'additional_items.clip_on_type' => 'nullable|string|max:191',
            'additional_items.clip_on_lens_type' => 'nullable|string|max:191',
            'additional_items.clip_on_lens_colour' => 'nullable|string|max:191',
            'additional_items.clip_on_lens_sub_colour' => 'nullable|string|max:191',
            'additional_items.repairs_services' => 'nullable|array',
            'additional_items.repairs_services.*' => 'nullable|string|max:191',
            'additional_items.repair_description' => 'nullable|string|max:500',

            // Misc
            'special_instructions' => 'nullable|string|max:1000',
            'is_multipair' => 'nullable|boolean',
            'multipair_ref' => 'nullable|string|max:191',
            'package' => 'nullable|string|max:191',

            'order_reference' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // 2. API Client Validation
        $apiClient = $request->get('api_client');
        if (!$apiClient) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid API client'
            ], 400);
        }

        // 3. Get User and User-Specific Pricing Info
        $user = \App\Models\Auth\User::where('ac_code', $request->account_number)->firstOrFail();
        $profile = $user->profile;
        $priceId = $profile->cust_prlist_default ?? 0;
        $discount = $profile->cust_vsp_office_code ?? 0;

        // 4. Validate Lens Code Combination with User-Specific Pricing
        $lensValidation = DB::table('rx_validations')
            ->where('material_code', 'LIKE', '%' . trim($request->material_code) . '%')
            ->where('lens_code', 'LIKE', '%' . trim($request->lens_code) . '%')
            ->where('colour_code', 'LIKE', '%' . trim($request->colour_code) . '%')
            ->where('vendor_code', 'LIKE', '%' . trim($request->vendor_code) . '%')
            ->where('coating_code', 'LIKE', '%' . trim($request->coating_code) . '%')
            ->first();

        if (!$lensValidation) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid lens code combination.',
                'errors' => [
                    'lens_combination' => ['The selected combination of lens codes is not valid.']
                ]
            ], 422);
        }

        // 5. Validate Package (if provided)
        if (!empty($request->package)) {
            $packageExists = DB::table('frame_packages')
                ->where('name', $request->package)
                ->where('status', 1)
                ->exists();
            
            if (!$packageExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid package selected.',
                    'errors' => [
                        'package' => ['The selected package is not available.']
                    ]
                ], 422);
            }
        }

        // 6. Validate RX parameters against min/max values from lensValidation
        $minMaxValues = [
            'sph' => ['min' => $lensValidation->sphMin ?? null, 'max' => $lensValidation->sphMax ?? null],
            'cyl' => ['min' => $lensValidation->cylMin ?? null, 'max' => $lensValidation->cylMax ?? null],
            'axis' => ['min' => $lensValidation->axisMin ?? null, 'max' => $lensValidation->axisMax ?? null],
            'add' => ['min' => $lensValidation->addMin ?? null, 'max' => $lensValidation->addMax ?? null],
            'pd' => ['min' => $lensValidation->dpdMin ?? null, 'max' => $lensValidation->dpdMax ?? null],
            'base_curve' => ['min' => null, 'max' => null], // Would need base curve validation logic
            'prism' => ['min' => $lensValidation->prism1Min ?? null, 'max' => $lensValidation->prism1Max ?? null],
        ];
        $rxValidationErrors = self::validateRxParameters($request->rx, $minMaxValues);
        if (!empty($rxValidationErrors)) {
            return response()->json([
                'success' => false,
                'message' => 'RX parameters validation failed',
                'errors' => $rxValidationErrors
            ], 422);
        }

        // 7. Calculate frontend-only fields (like JavaScript does)
        $calculatedData = $this->calculateFrontendFields($request->all(), $lensValidation);

        // 8. Calculate Price with User-Specific Logic (like form does)
        $calculatedPrice = $this->calculateOrderPriceWithUserLogic($lensValidation, $calculatedData, $priceId, $discount);

        // 9. Save Order
        DB::beginTransaction();
        try {
            $order = \App\Models\Backend\Order::create([
                'user_id' => $user->id,
                'type' => \App\Models\Backend\Order::ORDER_RX,
                'method' => 'W',
                'status' => \App\Models\Backend\OrderStatus::getStatus(\App\Models\Backend\OrderStatus::STATUS_SUBMITTED)->id,
                'price' => $calculatedPrice,
                'patient' => $request->first_name . ' ' . $request->last_name,

                // Save all codes for reporting/querying
                'material_code' => $request->material_code,
                'lens_code' => $request->lens_code,
                'colour_code' => $request->colour_code,
                'vendor_code' => $request->vendor_code,
                'coating_code' => $request->coating_code,
            ]);

            \App\Models\Backend\RxOrders::create([
                'order_id' => $order->id,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'data' => json_encode($calculatedData),
            ]);

            DB::commit();

            // 10. Return Complete Response
            return response()->json([
                'success' => true,
                'message' => 'Rx order created successfully',
                'data' => [
                    'order_id' => $order->id,
                    'order_number' => $order->getOrderNumber(),
                    'total_price' => $order->price,
                    'status' => $order->statusModel->name ?? 'Submitted',
                    'created_at' => $order->created_at,
                    'patient' => $order->patient,
                    'lens' => [
                        'lens_type' => $calculatedData['lens']['lens_type'] ?? null,
                        'lens_manufacturer' => $calculatedData['lens']['lens_manufacturer'] ?? null,
                        'lens_name' => $calculatedData['lens']['lens_name'] ?? null,
                        'lens_material' => $calculatedData['lens']['lens_material'] ?? null,
                        'lens_filter' => $calculatedData['lens']['lens_filter'] ?? null,
                        'lens_colour' => $calculatedData['lens']['lens_colour'] ?? null,
                        'coating_made' => $calculatedData['lens']['coating_made'] ?? null,
                        'coating_name' => $calculatedData['lens']['coating_name'] ?? null,
                        'coating_colour' => $calculatedData['lens']['coating_colour'] ?? null,
                        'sub_type' => $calculatedData['lens']['sub_type'] ?? null,
                        'base_curve' => $calculatedData['lens']['base_curve'] ?? null
                    ],
                    'codes' => [
                        'material_code' => $order->material_code,
                        'lens_code' => $order->lens_code,
                        'colour_code' => $order->colour_code,
                        'vendor_code' => $order->vendor_code,
                        'coating_code' => $order->coating_code,
                    ],
                    'right_eye' => [
                        'sphere' => $calculatedData['rx']['distance']['right_sph'] ?? null,
                        'cylinder' => $calculatedData['rx']['distance']['right_cyl'] ?? null,
                        'axis' => $calculatedData['rx']['distance']['right_axis'] ?? null,
                        'add' => $calculatedData['rx']['add_for_reading']['right_add'] ?? null,
                        'sphere_new' => $calculatedData['rx']['distance']['right_sph_new'] ?? null,
                    ],
                    'left_eye' => [
                        'sphere' => $calculatedData['rx']['distance']['left_sph'] ?? null,
                        'cylinder' => $calculatedData['rx']['distance']['left_cyl'] ?? null,
                        'axis' => $calculatedData['rx']['distance']['left_axis'] ?? null,
                        'add' => $calculatedData['rx']['add_for_reading']['left_add'] ?? null,
                        'sphere_new' => $calculatedData['rx']['distance']['left_sph_new'] ?? null,
                    ]
                ],
                'frame_type' => $calculatedData['frame']['frame_type'] ?? null,
                'frame_size' => $calculatedData['frame']['frame_size'] ?? null,
                'frame_info' => $calculatedData['frame']['frame_info'] ?? null,
                'brand' => $calculatedData['frame']['frame_info']['brand'] ?? null,
                'model' => $calculatedData['frame']['frame_info']['model'] ?? null,
                'colour' => $calculatedData['frame']['frame_info']['colour'] ?? null,
                'size' => $calculatedData['frame']['frame_info']['size'] ?? null,
                'pof' => $calculatedData['frame']['frame_info']['pof'] ?? null,
                'material_code' => $request->material_code,
                'lens_code' => $request->lens_code,
                'colour_code' => $request->colour_code,
                'vendor_code' => $request->vendor_code,
                'coating_code' => $request->coating_code,
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while creating the order',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Calculate frontend-only fields and ensure complete data structure (like JavaScript does)
     */
    protected function calculateFrontendFields($data, $lensValidation = null)
    {
        // Start with the original data
        $calculatedData = $data;

        // Ensure all required sections exist with proper structure
        $calculatedData = $this->ensureCompleteDataStructure($calculatedData);

        // Populate lens information from the validation lookup if available
        if ($lensValidation) {
            $calculatedData['lens']['lens_type'] = $lensValidation->lens_type ?? '';
            $calculatedData['lens']['sub_type'] = $lensValidation->sub_type ?? '';
            $calculatedData['lens']['lens_manufacturer'] = $lensValidation->lens_manufacturer ?? '';
            $calculatedData['lens']['lens_name'] = $lensValidation->lens_name ?? '';
            $calculatedData['lens']['base_curve'] = $lensValidation->base_curve ?? '';
            $calculatedData['lens']['lens_material'] = $lensValidation->lens_material ?? '';
            $calculatedData['lens']['lens_filter'] = $lensValidation->lens_filter ?? '';
            $calculatedData['lens']['lens_colour'] = $lensValidation->lens_colour ?? '';
            $calculatedData['lens']['coating_made'] = $lensValidation->coating_made ?? '';
            $calculatedData['lens']['coating_name'] = $lensValidation->coating_name ?? '';
            $calculatedData['lens']['coating_colour'] = $lensValidation->coating_colour ?? '';
        }

        // Calculate sph_new values for intermediate lens conversions
        if (isset($data['lens']['sub_type']) && $data['lens']['sub_type'] === 'Intermediate_(Lab_Converts_Rx)') {
            // Calculate for right eye
            if (!empty($data['rx']['distance']['right_sph']) && 
                !empty($data['rx']['add_for_reading']['right_add']) && 
                !empty($data['rx']['add_for_reading']['right_intermediate_strength'])) {
                
                $newSph = floatval($data['rx']['distance']['right_sph']) + 
                         (floatval($data['rx']['add_for_reading']['right_add']) * 
                          (floatval($data['rx']['add_for_reading']['right_intermediate_strength']) / 100));
                
                $calculatedData['rx']['distance']['right_sph_new'] = round($newSph * 4) / 4; // Round to nearest quarter
            }

            // Calculate for left eye
            if (!empty($data['rx']['distance']['left_sph']) && 
                !empty($data['rx']['add_for_reading']['left_add']) && 
                !empty($data['rx']['add_for_reading']['right_intermediate_strength'])) {
                
                $newSph = floatval($data['rx']['distance']['left_sph']) + 
                         (floatval($data['rx']['add_for_reading']['left_add']) * 
                          (floatval($data['rx']['add_for_reading']['right_intermediate_strength']) / 100));
                
                $calculatedData['rx']['distance']['left_sph_new'] = round($newSph * 4) / 4; // Round to nearest quarter
            }
        } else {
            // Regular add power conversion
            // Calculate for right eye
            if (!empty($data['rx']['distance']['right_sph']) && !empty($data['rx']['add_for_reading']['right_add'])) {
                $newSph = floatval($data['rx']['distance']['right_sph']) + floatval($data['rx']['add_for_reading']['right_add']);
                $calculatedData['rx']['distance']['right_sph_new'] = round($newSph * 4) / 4; // Round to nearest quarter
            }

            // Calculate for left eye
            if (!empty($data['rx']['distance']['left_sph']) && !empty($data['rx']['add_for_reading']['left_add'])) {
                $newSph = floatval($data['rx']['distance']['left_sph']) + floatval($data['rx']['add_for_reading']['left_add']);
                $calculatedData['rx']['distance']['left_sph_new'] = round($newSph * 4) / 4; // Round to nearest quarter
            }
        }

        // Calculate final RX values (copy from distance to final_rx)
        if (!empty($data['rx']['distance']['right_sph'])) {
            $calculatedData['rx']['final_rx']['right_sph'] = $data['rx']['distance']['right_sph'];
        }
        if (!empty($data['rx']['distance']['right_cyl'])) {
            $calculatedData['rx']['final_rx']['right_cyl'] = $data['rx']['distance']['right_cyl'];
        }
        if (!empty($data['rx']['distance']['right_axis'])) {
            $calculatedData['rx']['final_rx']['right_axis'] = $data['rx']['distance']['right_axis'];
        }
        if (!empty($data['rx']['distance']['left_sph'])) {
            $calculatedData['rx']['final_rx']['left_sph'] = $data['rx']['distance']['left_sph'];
        }
        if (!empty($data['rx']['distance']['left_cyl'])) {
            $calculatedData['rx']['final_rx']['left_cyl'] = $data['rx']['distance']['left_cyl'];
        }
        if (!empty($data['rx']['distance']['left_axis'])) {
            $calculatedData['rx']['final_rx']['left_axis'] = $data['rx']['distance']['left_axis'];
        }

        // Calculate package_label if package is selected
        if (!empty($data['package'])) {
            $calculatedData['package_label'] = $data['package'] . ' (Tier Default)';
        }

        // Calculate package_price if package is selected (this would need business logic)
        if (!empty($data['package'])) {
            // This would need to be implemented based on your package pricing logic
            $calculatedData['package_price'] = 0.00; // Placeholder
        }

        return $calculatedData;
    }

    /**
     * Ensure the data structure matches the frontend form structure
     */
    protected function ensureCompleteDataStructure($data)
    {
        // Ensure basic structure exists
        $data['price'] = $data['price'] ?? 0;
        $data['first_name'] = $data['first_name'] ?? '';
        $data['last_name'] = $data['last_name'] ?? '';
        $data['special_instructions'] = $data['special_instructions'] ?? '';
        $data['is_multipair'] = $data['is_multipair'] ?? false;
        $data['multipair_ref'] = $data['multipair_ref'] ?? '';
        $data['package'] = $data['package'] ?? '';
        $data['order_reference'] = $data['order_reference'] ?? '';

        // Ensure lens structure
        if (!isset($data['lens'])) {
            $data['lens'] = [];
        }
        $data['lens']['lens_type'] = $data['lens']['lens_type'] ?? '';
        $data['lens']['lens_type2'] = $data['lens']['lens_type2'] ?? '';
        $data['lens']['sub_type'] = $data['lens']['sub_type'] ?? '';
        $data['lens']['lens_manufacturer'] = $data['lens']['lens_manufacturer'] ?? '';
        $data['lens']['base_curve'] = $data['lens']['base_curve'] ?? '';
        $data['lens']['lens_material'] = $data['lens']['lens_material'] ?? '';
        $data['lens']['lens_filter'] = $data['lens']['lens_filter'] ?? '';
        $data['lens']['lens_colour'] = $data['lens']['lens_colour'] ?? '';
        $data['lens']['coating_made'] = $data['lens']['coating_made'] ?? '';
        $data['lens']['coating_name'] = $data['lens']['coating_name'] ?? '';
        $data['lens']['coating_colour'] = $data['lens']['coating_colour'] ?? '';
        $data['lens']['lens_name'] = $data['lens']['lens_name'] ?? '';

        // Ensure rx structure
        if (!isset($data['rx'])) {
            $data['rx'] = [];
        }

        // Distance section
        if (!isset($data['rx']['distance'])) {
            $data['rx']['distance'] = [];
        }
        $data['rx']['distance']['right_sph'] = $data['rx']['distance']['right_sph'] ?? '';
        $data['rx']['distance']['right_cyl'] = $data['rx']['distance']['right_cyl'] ?? '';
        $data['rx']['distance']['right_axis'] = $data['rx']['distance']['right_axis'] ?? '';
        $data['rx']['distance']['right_oc'] = $data['rx']['distance']['right_oc'] ?? '';
        $data['rx']['distance']['right_dist_type'] = $data['rx']['distance']['right_dist_type'] ?? '';
        $data['rx']['distance']['right_strength'] = $data['rx']['distance']['right_strength'] ?? '';
        $data['rx']['distance']['left_sph'] = $data['rx']['distance']['left_sph'] ?? '';
        $data['rx']['distance']['left_cyl'] = $data['rx']['distance']['left_cyl'] ?? '';
        $data['rx']['distance']['left_axis'] = $data['rx']['distance']['left_axis'] ?? '';
        $data['rx']['distance']['left_oc'] = $data['rx']['distance']['left_oc'] ?? '';
        $data['rx']['distance']['left_dist_type'] = $data['rx']['distance']['left_dist_type'] ?? '';
        $data['rx']['distance']['left_strength'] = $data['rx']['distance']['left_strength'] ?? '';

        // Prism section
        if (!isset($data['rx']['prism'])) {
            $data['rx']['prism'] = [];
        }
        $data['rx']['prism']['right_prism1'] = $data['rx']['prism']['right_prism1'] ?? '';
        $data['rx']['prism']['right_base1'] = $data['rx']['prism']['right_base1'] ?? '';
        $data['rx']['prism']['right_prism2'] = $data['rx']['prism']['right_prism2'] ?? '';
        $data['rx']['prism']['right_base2'] = $data['rx']['prism']['right_base2'] ?? '';
        $data['rx']['prism']['left_prism1'] = $data['rx']['prism']['left_prism1'] ?? '';
        $data['rx']['prism']['left_base1'] = $data['rx']['prism']['left_base1'] ?? '';
        $data['rx']['prism']['left_prism2'] = $data['rx']['prism']['left_prism2'] ?? '';
        $data['rx']['prism']['left_base2'] = $data['rx']['prism']['left_base2'] ?? '';

        // Add for reading section
        if (!isset($data['rx']['add_for_reading'])) {
            $data['rx']['add_for_reading'] = [];
        }
        $data['rx']['add_for_reading']['right_add'] = $data['rx']['add_for_reading']['right_add'] ?? '';
        $data['rx']['add_for_reading']['right_boc'] = $data['rx']['add_for_reading']['right_boc'] ?? '';
        $data['rx']['add_for_reading']['right_sh'] = $data['rx']['add_for_reading']['right_sh'] ?? '';
        $data['rx']['add_for_reading']['right_intermediate_strength'] = $data['rx']['add_for_reading']['right_intermediate_strength'] ?? '';
        $data['rx']['add_for_reading']['left_add'] = $data['rx']['add_for_reading']['left_add'] ?? '';
        $data['rx']['add_for_reading']['left_boc'] = $data['rx']['add_for_reading']['left_boc'] ?? '';
        $data['rx']['add_for_reading']['left_sh'] = $data['rx']['add_for_reading']['left_sh'] ?? '';
        $data['rx']['add_for_reading']['left_intermediate_strength'] = $data['rx']['add_for_reading']['left_intermediate_strength'] ?? '';

        // Extended info section
        if (!isset($data['rx']['extended_info'])) {
            $data['rx']['extended_info'] = [];
        }
        $data['rx']['extended_info']['right_pd'] = $data['rx']['extended_info']['right_pd'] ?? '';
        $data['rx']['extended_info']['right_intermediate_pd'] = $data['rx']['extended_info']['right_intermediate_pd'] ?? '';
        $data['rx']['extended_info']['right_npd'] = $data['rx']['extended_info']['right_npd'] ?? '';
        $data['rx']['extended_info']['right_base_curve'] = $data['rx']['extended_info']['right_base_curve'] ?? '';
        $data['rx']['extended_info']['right_oc'] = $data['rx']['extended_info']['right_oc'] ?? '';
        $data['rx']['extended_info']['left_pd'] = $data['rx']['extended_info']['left_pd'] ?? '';
        $data['rx']['extended_info']['left_intermediate_pd'] = $data['rx']['extended_info']['left_intermediate_pd'] ?? '';
        $data['rx']['extended_info']['left_npd'] = $data['rx']['extended_info']['left_npd'] ?? '';
        $data['rx']['extended_info']['left_base_curve'] = $data['rx']['extended_info']['left_base_curve'] ?? '';
        $data['rx']['extended_info']['left_oc'] = $data['rx']['extended_info']['left_oc'] ?? '';

        // Special info section
        if (!isset($data['rx']['special_info'])) {
            $data['rx']['special_info'] = [];
        }
        $data['rx']['special_info']['pento'] = $data['rx']['special_info']['pento'] ?? '';
        $data['rx']['special_info']['wrap'] = $data['rx']['special_info']['wrap'] ?? '';
        $data['rx']['special_info']['tilt'] = $data['rx']['special_info']['tilt'] ?? '';

        // Final RX section
        if (!isset($data['rx']['final_rx'])) {
            $data['rx']['final_rx'] = [];
        }
        $data['rx']['final_rx']['right_sph'] = $data['rx']['final_rx']['right_sph'] ?? '';
        $data['rx']['final_rx']['right_cyl'] = $data['rx']['final_rx']['right_cyl'] ?? '';
        $data['rx']['final_rx']['right_axis'] = $data['rx']['final_rx']['right_axis'] ?? '';
        $data['rx']['final_rx']['left_sph'] = $data['rx']['final_rx']['left_sph'] ?? '';
        $data['rx']['final_rx']['left_cyl'] = $data['rx']['final_rx']['left_cyl'] ?? '';
        $data['rx']['final_rx']['left_axis'] = $data['rx']['final_rx']['left_axis'] ?? '';

        // Frame section
        if (!isset($data['frame'])) {
            $data['frame'] = [];
        }
        $data['frame']['job_type'] = $data['frame']['job_type'] ?? '';
        $data['frame']['frame_type'] = $data['frame']['frame_type'] ?? '';
        $data['frame']['invoice_order_no'] = $data['frame']['invoice_order_no'] ?? '';
        // $data['frame']['ft_other'] = $data['frame']['ft_other'] ?? '';

        // Frame info
        if (!isset($data['frame']['frame_info'])) {
            $data['frame']['frame_info'] = [];
        }
        $data['frame']['frame_info']['brand'] = $data['frame']['frame_info']['brand'] ?? '';
        $data['frame']['frame_info']['model'] = $data['frame']['frame_info']['model'] ?? '';
        $data['frame']['frame_info']['colour'] = $data['frame']['frame_info']['colour'] ?? '';
        $data['frame']['frame_info']['size'] = $data['frame']['frame_info']['size'] ?? '';

        // Frame size
        if (!isset($data['frame']['frame_size'])) {
            $data['frame']['frame_size'] = [];
        }
        $data['frame']['frame_size']['a'] = $data['frame']['frame_size']['a'] ?? '';
        $data['frame']['frame_size']['b'] = $data['frame']['frame_size']['b'] ?? '';
        $data['frame']['frame_size']['ed'] = $data['frame']['frame_size']['ed'] ?? '';
        $data['frame']['frame_size']['dbl'] = $data['frame']['frame_size']['dbl'] ?? '';

        // Additional items
        if (!isset($data['additional_items'])) {
            $data['additional_items'] = [];
        }
        $data['additional_items']['clip_on_type'] = $data['additional_items']['clip_on_type'] ?? '';
        $data['additional_items']['clip_on_lens_type'] = $data['additional_items']['clip_on_lens_type'] ?? '';
        $data['additional_items']['clip_on_lens_colour'] = $data['additional_items']['clip_on_lens_colour'] ?? '';
        $data['additional_items']['clip_on_lens_sub_colour'] = $data['additional_items']['clip_on_lens_sub_colour'] ?? '';
        $data['additional_items']['repairs_services'] = $data['additional_items']['repairs_services'] ?? [];
        $data['additional_items']['repair_description'] = $data['additional_items']['repair_description'] ?? '';

        return $data;
    }

    /**
     * Calculate the Rx order price based on the validated lens and request data.
     * Implement your business logic here.
     */
    protected function calculateOrderPrice($lensValidation, $data)
    {
        $price = floatval($lensValidation->price);

        // Add package price if selected
        if (!empty($data['package_price'])) {
            $price += floatval($data['package_price']);
        }

        // Add-ons, repairs, etc. (implement as needed)
        // $price += ...;

        return $price;
    }

    /**
     * Calculate the Rx order price based on the validated lens and request data,
     * including user-specific pricing, package pricing, and add-ons pricing.
     */
    protected function calculateOrderPriceWithUserLogic($lensValidation, $data, $priceId, $discount)
    {
        // Get user-specific price from lens validation
        $priceKey = 'Price_' . $priceId;
        $basePrice = 0.00;
        
        // Find the correct price field for this user's price list
        if (isset($lensValidation->$priceKey)) {
            $basePrice = floatval($lensValidation->$priceKey);
        } else {
            // Fallback to default price if user-specific price not found
            $basePrice = floatval($lensValidation->price ?? 0);
        }

        $packagePrice = 0.00;
        $addOnsPrice = 0.00;
        $framePrice = 0.00;

        // Add package price if selected
        if (!empty($data['package'])) {
            $package = DB::table('frame_packages')
                ->where('name', $data['package'])
                ->where('status', 1)
                ->first();
            if ($package) {
                $packagePrice = floatval($package->price ?? 0);
            }
        }

        // Calculate add-ons price
        if (!empty($data['additional_items'])) {
            // Clip-on pricing
            if (!empty($data['additional_items']['clip_on_type'])) {
                $addOnsPrice += 15.00; // Example clip-on price
            }
            
            // Repairs/services pricing
            if (!empty($data['additional_items']['repairs_services'])) {
                foreach ($data['additional_items']['repairs_services'] as $service) {
                    // This would require a lookup table for service prices
                    // For now, using a simple pricing model
                    $addOnsPrice += 10.00; // Example: $10 per service
                }
            }
        }

        // Calculate frame price (if frame info is provided)
        // if (!empty($data['frame']['frame_info']['brand']) && !empty($data['frame']['frame_info']['model'])) {
        //     $frameStyle = DB::table('frame_styles')
        //         ->where('brand_name', $data['frame']['frame_info']['brand'])
        //         ->where('name', $data['frame']['frame_info']['model'])
        //         ->where('color', $data['frame']['frame_info']['colour'] ?? '')
        //         ->where('size', $data['frame']['frame_info']['size'] ?? '')
        //         ->first();
            
        //     if ($frameStyle && !empty($frameStyle->price)) {
        //         $framePrice = floatval($frameStyle->price);
        //     }
        // }

        // Apply discount if applicable
        $totalPrice = $basePrice + $packagePrice + $addOnsPrice + $framePrice;
        
        if ($discount > 0) {
            $totalPrice *= (1 - ($discount / 100));
        }

        return round($totalPrice, 2);
    }

    /**
     * Validate RX parameters against min/max values from lensValidation
     */
    private static function validateRxParameters($rx, $minMaxValues)
    {
        $errors = [];

        foreach ($rx as $key => $value) {
            if (is_array($value)) {
                foreach ($value as $subKey => $subValue) {
                    if (isset($minMaxValues[$subKey])) {
                        $min = $minMaxValues[$subKey]['min'];
                        $max = $minMaxValues[$subKey]['max'];
                        
                        // Only validate if min/max values are not null
                        if ($min !== null && $max !== null) {
                            if ($subValue < $min || $subValue > $max) {
                                $errors[$key][$subKey] = "The value for $subKey is out of range. It should be between $min and $max.";
                            }
                        }
                    }
                }
            } else {
                if (isset($minMaxValues[$key])) {
                    $min = $minMaxValues[$key]['min'];
                    $max = $minMaxValues[$key]['max'];
                    
                    // Only validate if min/max values are not null
                    if ($min !== null && $max !== null) {
                        if ($value < $min || $value > $max) {
                            $errors[$key] = "The value for $key is out of range. It should be between $min and $max.";
                        }
                    }
                }
            }
        }

        return $errors;
    }
} 