<?php

namespace App\Http\Controllers\Backend\Auth;

use Illuminate\Http\Request;
use App\Models\Backend\Order;
use App\Models\Backend\Notification;
use App\Models\Backend\EmailTemplate;
use App\Events\Backend\Auth\OrderStatusMail;
use App\Events\Frontend\Auth\CustomerReplied;
use App\Http\Requests\Backend\Auth\ShowEmailTemplateRequest;
use App\Http\Requests\Backend\Auth\StoreEmailTemplateRequest;
use App\Http\Requests\Backend\Auth\UpdateEmailTemplateRequest;
use App\Http\Requests\Backend\Auth\ManageEmailTemplateRequest;

use App\Http\Controllers\Controller;

class EmailTemplateController extends Controller
{

    /**
     * @var Request
     */
    protected $request;


    /**
     * @var Notification
     */
    protected $notification;

    /**
     * Notification constructor.
     *
     * @param Notification $notification
     */
    public function __construct(Request $request, Notification $notification)
    {
        $this->request = $request;
        $this->notification = $notification;
    }

    /**
     * Display a listing of the resource.
     * 
     * @param ShowEmailTemplateRequest $request
     * @return \Illuminate\Http\Response
     */
    public function index(ShowEmailTemplateRequest $request)
    {
        return view('backend.auth.templates.index')
            ->withTemplates(EmailTemplate::all());
    }

    /**
     * Show the form for creating a new resource.
     * 
     * @param ManageEmailTemplateRequest $request
     *
     * @return \Illuminate\Http\Response
     */
    public function create(ManageEmailTemplateRequest $request)
    {
        // return view('backend.auth.templates.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreEmailTemplateRequest $request)
    {
        $template = EmailTemplate::create($request->all());
        if ($template){
            return redirect()->route('admin.auth.template.index')->withFlashSuccess(__('alerts.backend.templates.created'));
        } else {
            throw new GeneralException(__('exceptions.backend.access.templates.create_error'));
        }
    }
    
    /**
     * Display the specified resource.
     *
     * @param  ShowEmailTemplateRequest  $request
     * @param  EmailTemplate  $template
     * 
     * @return \Illuminate\Http\Response
     */
    public function show(ShowEmailTemplateRequest $request, EmailTemplate $template)
    {
        if ($request->ajax()) {
            return view('backend.auth.templates.includes.show-template')
                ->withMessage(EmailTemplate::find($request->id));            
        } else {
            return view('backend.auth.templates.show')
            ->withTemplate($template);
        }
    }

    /**
     * Show the form for editing the specified resource.
     * 
     * @param  ManageEmailTemplateRequest  $request
     * @param  EmailTemplate  $template
     * 
     * @return \Illuminate\Http\Response
     */
    public function edit(ManageEmailTemplateRequest $request, EmailTemplate $template)
    {
        return view('backend.auth.templates.edit')
            ->withTemplate($template);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\UpdateEmailTemplateRequest  $request
     * @param  EmailTemplate  $template
     * @return mixed
     */
    public function update(UpdateEmailTemplateRequest $request, EmailTemplate $template)
    {
        $model = $template->update($request->all());
        if ($model) {
            return redirect()->route('admin.auth.template.index')->withFlashSuccess(__('alerts.backend.templates.updated'));
        } else {
            throw new GeneralException(__('exceptions.backend.access.templates.update_error'));
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  ManageEmailTemplateRequest  $request
     * @param  EmailTemplate  $template
     * 
     * @return mixed
     */
    public function destroy(ManageEmailTemplateRequest $request, EmailTemplate $template)
    {
        if ($template->deleteTemplate($template)){
            return redirect()->route('admin.auth.template.index')->withFlashSuccess(__('alerts.backend.templates.deleted'));
        } else {
            throw new GeneralException(__('exceptions.backend.access.templates.delete_error'));
        }
    }

    /**
     * Send default order status email to customer .
     *
     * @param  Request  $request
     * 
     * @return bool
     */
    public function sendMail(Request $request) {

        if ($request->ajax()) {

            $notification = new Notification;
            $email = EmailTemplate::find($request->template_id);
            $order = Order::find($request->order_id);

            // Admin sends an email to customer against the 
            // query made by customer about specific order
            event(new OrderStatusMail($email,$order));

            // A new notification sent to the customer  
            // after email by admin to customer 

            $this->notification->sendToCustomer($order,$email);
        }
    }

    /**
     * reply to admin from customer .
     *
     * @param  Request  $request
     * 
     * @return bool
     */
    public function reply(){

        event(new CustomerReplied($this->request->all()));

        $email = EmailTemplate::find($this->request->temp_id);

        $order = Order::find($this->request->order_id);

        $this->notification->sendToAdmin($order, $email,$this->request);
        
        return redirect()->route('login');
    }
}
