<?php

namespace App\Http\Controllers\Backend\Auth\User;

use App\Models\Auth\Role;
use App\Models\Auth\User;
use Illuminate\Support\Collection;
use App\Models\Backend\UserProfile;
use App\Http\Controllers\Controller;
use App\Models\Backend\CustomerLinkStaff;
use App\Events\Backend\Auth\User\UserDeleted;
use App\Events\Backend\Auth\User\UserCreated;
use App\Models\Backend\ImportedConsumablesOrder;
use App\Repositories\Backend\Auth\RoleRepository;
use App\Repositories\Backend\Auth\UserRepository;
use App\Repositories\Backend\Auth\PermissionRepository;
use App\Http\Requests\Backend\Auth\User\ShowUserRequest;
use App\Events\Backend\Auth\csvCustomersUploadingReport;
use App\Http\Requests\Backend\Auth\User\StoreUserRequest;
use App\Http\Requests\Backend\Auth\User\ManageUserRequest;
use App\Http\Requests\Backend\Auth\User\UpdateUserRequest;

/**
 * Class UserController.
 */
class UserController extends Controller
{
    /**
     * @var UserRepository
     */
    protected $userRepository;

    /**
     * UserController constructor.
     *
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(ShowUserRequest $request, User $user)
    {
        // Get the authenticated user
        $authUser = auth()->user();
        // Redirect lab managers/staff to orders list
        if ($authUser->isLabManager() || $authUser->isLabStaff()) {
            return redirect()->route('admin.orders.list');
        }
        $pageNo = $request->get('page-no', 1);
        $recordsPerPage = config('access.page_size');

        $model = $this->userRepository->getActivePaginated($recordsPerPage, 'ac_code', 'asc',$request->all(), $pageNo);
        if($request->ajax()){
            return view('partials.user-table')->withUsers($model)->withPageCount($pageNo);
        }
        // Get total records for initial load
        $tempModel = $this->userRepository->getActivePaginated(1, 'ac_code', 'asc', $request->all(), 1);
        $totalRecords = $tempModel->total();
        return view('backend.auth.user.index')
            ->withUsers($model)->withPageCount($pageNo)->withTotalRecords($totalRecords);
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function search(ShowUserRequest $request)
    {
        if($request->ajax()){
            $pageNo = $request->get('page-no', 1);
            $recordsPerPage = $request->get('records-per-page', config('access.list_page_size'));

            $model = $this->userRepository->getActivePaginated($recordsPerPage, 'ac_code', 'asc', $request->all(), $pageNo);
            $html = view('partials.user-table')->withUsers($model)->withPageCount($pageNo)->render();
            return response()->json([
                'html' => $html,
                'totalRecords' => $model->total(),
                'loadedRecords' => $model->count()
            ]);
        }
    }

    /**
     * @param ManageUserRequest    $request
     * @param RoleRepository       $roleRepository
     * @param PermissionRepository $permissionRepository
     *
     * @return mixed
     */
    public function create(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository)
    {
        $roles = $roleRepository->crudRoles();
        return view('backend.auth.user.create')
            ->withRoles($roles);
    }

    /**
     * @param StoreUserRequest  $request
     * @param User              $user
     * @param Role              $role
     *
     * @throws \Throwab le
     * @return mixed
     */
    public function store(StoreUserRequest $request, User $user, Role $role)
    {
        $user = $this->userRepository->create($request->all());
        return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.created'));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @return mixed
     */
    public function show(ShowUserRequest $request, User $user)
    {

        $customerStaff = null;
        if ($user->isCustomerAdmin()) {
            $customerStaff = $this->userRepository->getCustomerStaff($user);
        }

        return view('backend.auth.user.show')
            ->withUser($user)
            ->withProfile($user->profile)
            ->withCustomerStaff($customerStaff);
    }

    /**
     * @param PermissionRepository $permissionRepository
     * @param RoleRepository       $roleRepository
     * @param ManageUserRequest    $request
     * @param User                 $user
     *
     * @return mixed
     */
    public function edit(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository, User $user)
    {
        $profile = new UserProfile();
        if(!empty($user->userProfile))
            $profile = $user->userProfile;

        return view('backend.auth.user.edit')
            ->withUser($user)
            ->withProfile($profile)
            ->withRoles($roleRepository->get())
            ->withUserRoles($user->roles->pluck('name')->all());
    }

    /**
     * @param UpdateUserRequest $request
     * @param User              $user
     *
     * @throws \App\Exceptions\GeneralException
     * @throws \Throwable
     * @return mixed
     */
    public function update(UpdateUserRequest $request, User $user)
    {
        $this->userRepository->update($user, $request->all());

        return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.updated'));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @throws \Exception
     * @return mixed
     */
    public function destroy(ManageUserRequest $request, User $user)
    {
        $this->userRepository->forceDelete($user);

        event(new UserDeleted($user));

        return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.deleted'));
    }

    /**
     * @param UpdateProfileRequest $request
     *
     * @throws \App\Exceptions\GeneralException
     * @return mixed
     */
    public function updateApi(ManageUserRequest $request, User $user)
    {
        \Log::info('updateApi called', [
            'user_id' => $user->id,
            'steps' => $request->get('steps'),
            'has_file' => $request->hasFile('avatar_location'),
            'file_info' => $request->hasFile('avatar_location') ? $request->file('avatar_location')->getClientOriginalName() : null,
            'all_data' => $request->all()
        ]);

        // Force immediate log write
        file_put_contents(storage_path('logs/laravel.log'), '[' . now() . '] local.INFO: updateApi called - user_id: ' . $user->id . ', steps: ' . $request->get('steps') . ', has_file: ' . ($request->hasFile('avatar_location') ? 'true' : 'false') . PHP_EOL, FILE_APPEND);

        if($request->get('steps') == 'personal_info'){
            $post = $request->all();
            $post['email'] = $user->email;
              // Handle name fields - split if full name is provided
            $post['first_name'] = $request->get('first_name', $user->first_name);
            $post['last_name'] = $request->get('last_name', $user->last_name);
            if(empty($post['active']))
                $post['active'] = 0;

            if(!isset($post['religion']))
                $post['religion'] = null;

            $output = $this->userRepository->update(
                $user,
                $post
            );
            $this->userRepository->updateAvatar(
                $user,
                $request->all(),
                $request->has('avatar_location') ? $request->file('avatar_location') : false
            );
        }elseif($request->get('steps') == 'avatar'){
            $output = $this->userRepository->updateAvatar(
                $user,
                $request->all(),
                $request->has('avatar_location') ? $request->file('avatar_location') : false
            );
            return response()->json(['message'=>__('strings.frontend.user.profile_updated'),'image'=>$user->picture]);
        }elseif($request->get('steps') == 'settings'){
            $output = $this->userRepository->updateSettings(
                $user,
                $request->all()
            );
        }else{
            return response()->json(['status'=>true,'message'=>'Profile is not updated!']);
        }
        return response()->json(['status'=>true,'message'=>__('strings.frontend.user.profile_updated'),'image'=>$user->picture,'active'=>$user->status_label]);
    }
}
