<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Process;
use App\Models\Backend\Cron;
use App\Services\CronService;
use Illuminate\Console\Scheduling\Schedule;

/**
 * Class CronController.
 */
class CronController extends Controller
{
    protected CronService $cronService;

    public function __construct(CronService $cronService)
    {
        $this->cronService = $cronService;
    }

    public function showNextRunTime(string $cronExpression): JsonResponse
    {
        $cronServiceResponse = $this->cronService->getNextScheduledRun($cronExpression);
        return $cronServiceResponse;
    }

    public function index(): JsonResponse
    {
        $process = Process::run(['crontab', '-l']);
        $cronJobs = explode("\n", trim($process->output()));
        $this->read($cronJobs);
        $crons = Cron::all()->toArray();
        return response()->json(['crons' => $crons]);
    }

    protected function read(array $cronJobs): void
    {
        unset($cronJobs[0], $cronJobs[1]);
        $artisanScheduleRunExists = collect($cronJobs)->contains(fn($job) => str_contains($job, 'artisan schedule:run'));

        if ($artisanScheduleRunExists) {
            // In Laravel 11+, get the schedule from the container
            $schedule = app(Schedule::class);

            foreach ($schedule->events() as $event) {
                $commandExists = collect($cronJobs)->contains(fn($job) => str_contains($job, $event->command));
                if (!$commandExists) {
                    $cronJobs[] = $event->expression . ' ' . $event->command;
                }
            }
        }

        $excludedCommands = [
            '/usr/local/bin/php /home/shapers/order.lens-shapers.com/artisan schedule:run >> /dev/null 2>&1',
            '/usr/local/bin/php /home/shapers/testing2.lens-shapers.com/artisan cron:scheduled >> /dev/null 2>&1'
        ];

        foreach ($cronJobs as $cronJob) {
            $parts = explode(' ', $cronJob, 6);
            if (count($parts) < 6) {
                continue;
            }
            [$minute, $hour, $day, $month, $weekday, $command] = $parts;

            if (in_array($command, $excludedCommands) || empty(trim($command))) {
                continue;
            }

            $cronExpression = "$minute $hour $day $month $weekday";
            $cronServiceJson = $this->showNextRunTime($cronExpression);
            $data = $cronServiceJson->getData(true);

            $isCommented = str_starts_with(trim($command), '#');
            $isActive = !$isCommented;

            $cronName = $isActive ? Cron::extractCronName($command) : 'Disabled Cron';
            $cron = Cron::firstOrNew(['command' => $command]);

            if (isset($data['description'])) {
                $cron->description = $data['description'];
            }

            $cron->fill([
                'command' => $command ?: 'Custom Cron',
                'description' => '',
                'name' => $cronName,
                'short_description' => $isActive ? ($data['short_description'] ?? '') : '-',
                'next_scheduled' => $isActive ? ($data['nextRun'] ?? '') : '-',
                'status' => $isCommented ? 2 : 1,
                'minute' => $minute,
                'hour' => $hour,
                'day' => $day,
                'month' => $month,
                'weekday' => $weekday,
                'updated_at' => now()->format('Y-m-d H:i:s'),
            ]);

            if (!$cron->exists) {
                $cron->created_at = now()->format('Y-m-d H:i:s');
            }

            $cron->save();
        }
    }

    public function getLogs(Request $request): JsonResponse
    {
        $cronJob = Cron::select('logs')->where('id', $request->id)->first();
        return response()->json(['message' => $cronJob]);
    }

    public function saveDescription(Request $request): JsonResponse
    {
        $request->validate([
            'id' => 'required|integer|exists:crons,id',
            'description' => 'required|string|max:255',
        ]);

        $cronJob = Cron::find($request->id);
        $cronJob->description = $request->description;
        $cronJob->save();

        return response()->json(['status' => 'success']);
    }
}