<?php

namespace App\Http\Controllers;

use App\Models\Backend\FrameBrands;
use App\Models\Backend\FrameManufacturer;
use App\Models\Backend\Package;
use App\Models\Backend\FrameStyles;
use App\Models\Backend\Cron;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Client;
use Exception;
use DB;
use League\Csv\Reader;

/**
 * Class FrameStyleController.
 */
class FrameStyleController extends Controller
{
    protected FrameBrands $brand;
    private string $ApiBaseUrl;
    private string $partnerId;
    private string $username;
    private string $szipcode;
    private string $location;
    private ?string $authToken;

    const MARKET = 'USA';
    const STATUS = 'A';

    public function __construct()
    {
        $this->brand = new FrameBrands();
        $this->ApiBaseUrl = env('FRAME_BASE_URL', 'http://api.framesdata.com:80/api');
        $this->partnerId = env('FRAME_PARTNERID', '87770985-822F-40AC-9822-CF061350B0B5');
        $this->username = env('FRAME_USERNAME', 'KA0699592');
        $this->szipcode = env('FRAME_SZIPCODE', 'L4W2A1');
        $this->location = env('FRAME_LOCATION', '1');
        $this->authToken = $this->getFrameAuthToken();
    }

    /**
     * Function to get time when was cron run last time to sync the frames records
     */
    private function cronScheduled(): ?string
    {
        $cronScheduled = Cron::select('last_scheduled', 'next_scheduled')
            ->where('name', 'syncFrameData:run')
            ->first();

        if (is_null($cronScheduled->last_scheduled) || empty($cronScheduled->last_scheduled)) {
            $nextScheduled = new \DateTime($cronScheduled->next_scheduled);
            $lastScheduled = $nextScheduled->sub(new \DateInterval('P7D'))->format('Y-m-d H:i:s');
            $cronScheduled->last_scheduled = $lastScheduled;
        }

        return $cronScheduled->last_scheduled;
    }

    /**
     * Function to render the frames table grid
     */
    public function index(Request $request): JsonResponse
    {
        if (!$request->ajax()) {
            return response()->json(['error' => 'Invalid request'], 400);
        }

        $filters = [
            'manufacturer' => $request->get('manufacturer'),
            'brand' => $request->get('brand'),
            'model' => $request->get('model'),
            'package' => $request->get('package'),
            'price' => $request->get('price'),
            'status' => $request->get('status'),
        ];

        $query = FrameStyles::select([
            'frame_styles.id',
            'frame_styles.name',
            'frame_styles.price',
            'frame_styles.tier',
            'frame_styles.charge',
            'frame_styles.model_id',
            'frame_styles.configuration_fpc',
            'frame_styles.status',
            'frame_styles.manufacturer_id',
            'frame_styles.brand_id',
            'frame_styles.package_data',
            'frame_styles.non_rx',
            'frame_styles.out_edge',
            'frame_styles.A',
            'frame_styles.B',
            'frame_styles.ED',
            'frame_styles.DBL',
            'frame_style_data',
        ]);

        $this->applyFilters($query, $filters);

        $manufacturersCollection = Cache::remember('manufacturers', 3600, function () {
            return FrameManufacturer::select('manufacturer_id', 'manufacturer_name')
                ->orderBy('manufacturer_name')
                ->get()
                ->toArray();
        });

        $brandsCollection = $this->getBrandsCollection($filters['manufacturer']);
        $modelsCollection = $filters['brand']
            ? $this->getModelsCollection($filters['brand'])
            : [];

        $responseData = [
            'packageCollection' => FrameStyles::packageCollection(),
            'chargeCollection' => FrameStyles::chargeCollection(),
            'tierCollection' => FrameStyles::tierCollection(),
            'priceFilters' => FrameStyles::priceFilters(),
            'statusFilter' => FrameStyles::statusFilter(),
            'manufacturersCollection' => $manufacturersCollection,
            'brandsCollection' => $brandsCollection,
            'modelsCollection' => $modelsCollection,
            'cronStamp' => self::cronScheduled(),
            'selectedManufacturerId' => $request->get('manufacturer'),
            'selectedBrandId' => $request->get('brand'),
        ];

        $totalFilteredCount = $query->count();
        $responseData['totalFilteredCount'] = $totalFilteredCount;

        if ($request->get('export')) {
            $responseData['stylesCollection'] = $query->get();
            return response()->json($responseData);
        }

        if (array_filter($filters)) {
            $stylesCollection = $query->paginate(100);
            $responseData['stylesCollection'] = $stylesCollection;
            $responseData['paginationLinks'] = (string) $stylesCollection->links();
            $responseData['count'] = $totalFilteredCount;
            $responseData['columns'] = $this->getColumnDefinitions();
        } else {
            $responseData['stylesCollection'] = [];
            $responseData['paginationLinks'] = [];
            $responseData['currentPageCount'] = 0;
            $responseData['count'] = 0;
            $responseData['columns'] = [];
        }

        if ($request->get('load_header')) {
            $responseData['headerHtml'] = view('partials.frame_header', [
                'count' => $totalFilteredCount,
                'packageCollection' => FrameStyles::packageCollection(),
                'priceFilters' => FrameStyles::priceFilters(),
                'statusFilter' => FrameStyles::statusFilter(),
                'manufacturersCollection' => $manufacturersCollection,
                'brandsCollection' => $brandsCollection,
                'modelsCollection' => $modelsCollection,
                'selectedManufacturerId' => $request->get('manufacturer'),
                'selectedBrandId' => $request->get('brand'),
                'selectedModelName' => $request->get('model'),
                'selectedPackageId' => $request->get('package'),
                'selectedPriceFilterId' => $request->get('price'),
                'selectedStatusFilterId' => $request->get('status'),
                'frameGroupingActive' => $request->get('frameGrouping', false),
                'cronStamp' => self::cronScheduled(),
            ])->render();
        }

        return response()->json($responseData);
    }

    /**
     * Filter function
     */
    protected function applyFilters($query, array $filters): void
    {
        if ($filters['manufacturer'] && $filters['manufacturer'] !== 'all') {
            $query->where('frame_styles.manufacturer_id', $filters['manufacturer']);
        }

        if ($filters['brand']) {
            $query->where('frame_styles.brand_id', $filters['brand']);
        }

        if ($filters['package']) {
            $query->where('frame_styles.package_data', 'like', '%"' . $filters['package'] . '":%');
        }

        if ($filters['price']) {
            if ($filters['price'] === 'non-blank') {
                $query->whereNotNull('frame_styles.price')
                    ->where('frame_styles.price', '<>', '');
            } elseif ($filters['price'] === 'blank') {
                $query->whereNull('frame_styles.price')
                    ->orWhere('frame_styles.price', '');
            }
        }

        if ($filters['status']) {
            if ($filters['status'] === 'A') {
                $query->where('frame_styles.status', 'A');
            } elseif ($filters['status'] === 'D') {
                $query->where('frame_styles.status', 'D');
            } elseif ($filters['status'] === 'All') {
                $query->whereIn('frame_styles.status', ['A', 'D']);
            }
        }

        if ($filters['model']) {
            $query->where('frame_styles.name', 'like', '%' . $filters['model'] . '%');
        }
    }

    /**
     * Get all brands collection and render to header with caching
     */
    protected function getBrandsCollection(?string $manufacturerId): array
    {
        if ($manufacturerId === 'all') {
            return Cache::remember('all_brands', 3600, function () {
                return FrameBrands::orderBy('brand_name')
                    ->get()
                    ->toArray();
            });
        }

        $cacheKey = $manufacturerId ? 'brands_' . $manufacturerId : 'brands';
        return Cache::remember($cacheKey, 3600, function () use ($manufacturerId) {
            $query = FrameBrands::query();
            if ($manufacturerId) {
                $query->where('manufacturer_id', $manufacturerId);
            }
            return $query->orderBy('brand_name')
                ->get()
                ->toArray();
        });
    }

    /**
     * Render model collection by selected brand id
     */
    protected function getModelsCollection(string $brandId): array
    {
        return Cache::remember("models_for_brand_{$brandId}", 300, function () use ($brandId) {
            return FrameStyles::where('brand_id', $brandId)
                ->select('model_id', 'name')
                ->distinct()
                ->orderBy('name')
                ->get()
                ->toArray();
        });
    }

    /**
     * Load table head th
     */
    protected function getColumnDefinitions(): array
    {
        return [
            ['id' => 'toggle', 'name' => '  '],
            ['width' => '5%', 'id' => 'packages', 'name' => 'Packages'],
            ['id' => 'tier', 'name' => 'Tier'],
            ['id' => 'price', 'name' => 'Price'],
            ['id' => 'charge', 'name' => 'Charge'],
            ['id' => 'non_rx', 'name' => 'Non-Rx'],
            ['id' => 'out_edge', 'name' => 'Out-Edge'],
            ['id' => 'manufacturer', 'name' => 'Manufacturer'],
            ['id' => 'brand', 'name' => 'Brand'],
            ['id' => 'model', 'name' => 'Model'],
            ['id' => 'frameColor', 'name' => 'Frame Color'],
            ['id' => 'size', 'name' => 'Size'],
            ['id' => 'frameCode', 'name' => 'Frame Code'],
            ['id' => 'a', 'name' => 'A'],
            ['id' => 'b', 'name' => 'B'],
            ['id' => 'ed', 'name' => 'ED'],
            ['id' => 'dbl', 'name' => 'DBL'],
            ['id' => 'edAngle', 'name' => 'EDAngle'],
            ['id' => 'rim', 'name' => 'RIM'],
            ['id' => 'edgeType', 'name' => 'Edge Type'],
            ['id' => 'image', 'name' => 'Image'],
            ['id' => 'configurationFpc', 'name' => 'Configuration FPC'],
            ['id' => 'upc', 'name' => 'UPC'],
            ['id' => 'sku', 'name' => 'SKU'],
            ['id' => 'status', 'name' => 'Status'],
            ['id' => 'material', 'name' => 'Material'],
            ['id' => 'preciousMetal', 'name' => 'Precious Metal'],
            ['id' => 'frameShape', 'name' => 'Frame Shape'],
            ['id' => 'rx', 'name' => 'RX'],
            ['id' => 'frameType', 'name' => 'Frame Type'],
            ['id' => 'lensColor', 'name' => 'Lens Color'],
            ['id' => 'lensColorCode', 'name' => 'Lens Color Code']
        ];
    }

    public function getFrameAuthToken(): ?string
    {
        $client = new Client();
        $url = $this->ApiBaseUrl . '/authenticatepms?partnerid=' . $this->partnerId . '&username=' . $this->username . '&szipcode=' . $this->szipcode . '&bzipcode=' . $this->szipcode . '&locations=' . $this->location;

        try {
            $response = $client->request('GET', $url);
            if ($response->getStatusCode() == 200) {
                $data = json_decode($response->getBody(), true);
                return $data['auth']['AuthorizationTicket'] ?? null;
            }
        } catch (Exception $e) {
            return null;
        }

        return null;
    }

    /**
     * Main function to sync frame data to database
     */
    public function syncFramesData(): JsonResponse
    {
        $client = new Client();

        try {
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '-1');
            $this->brandData($client);
            return response()->json(['message' => 'All frames data saved successfully!']);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'Unable to fetch records from api', 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Function to save brand data
     */
    public function brandData(Client $client): bool
    {
        $url = $this->ApiBaseUrl . '/brands?auth=' . $this->authToken;

        try {
            $response = $client->request('GET', $url);
            if ($response->getStatusCode() == 200) {
                $data = json_decode($response->getBody(), true);
                foreach ($data['Brands'] as $item) {
                    FrameBrands::updateOrCreate(
                        ['brand_id' => $item['BrandFramesMasterID']],
                        [
                            'manufacturer_id' => $item['ManufacturerFramesMasterID'],
                            'brand_id' => $item['BrandFramesMasterID'],
                            'brand_name' => $item['BrandName'],
                            'market' => $item['Market'],
                            'status' => $item['Status'],
                            'created_at' => now()
                        ]
                    );
                    $this->styleData($client, $item['BrandFramesMasterID']);
                }
                return true;
            }
        } catch (Exception $e) {
            throw $e;
        }

        return false;
    }

    /**
     * Function to save style data
     */
    public function styleData(Client $client, int $brand_id): bool
    {
        $url = $this->ApiBaseUrl . '/brands/' . $brand_id . '/styles?auth=' . $this->authToken;

        try {
            $response = $client->request('GET', $url);
            if ($response->getStatusCode() == 200) {
                $data = json_decode($response->getBody(), true);
                foreach ($data['Styles'] as $item) {
                    $frameSizeResponse = $this->getFrameSize($client, $item['StyleFramesMasterID']);
                    $arrayCount = count($frameSizeResponse);
                    foreach ($frameSizeResponse as $key => $value) {
                        if ($arrayCount != $key) {
                            try {
                                $frameSizeData = json_decode($value, true);
                                $mergedData = array_merge($item, $frameSizeData);
                                $jsonItemsWithSizeData = json_encode($mergedData);

                                FrameStyles::updateOrCreate(
                                    ['configuration_fpc' => $frameSizeData['ConfigurationFPC']],
                                    [
                                        'model_id' => $item['StyleFramesMasterID'],
                                        'manufacturer_id' => $item['ManufacturerFramesMasterID'],
                                        'brand_id' => $item['BrandFramesMasterID'],
                                        'collection_id' => $item['CollectionFramesMasterID'],
                                        'color' => $frameSizeData['FrameColor'],
                                        'configuration_fpc' => $frameSizeData['ConfigurationFPC'],
                                        'upc' => $frameSizeData['UPC'],
                                        'size' => $frameSizeData['EyeSize'] . '-' . $frameSizeData['BridgeSize'] . '-' . $frameSizeData['TempleLength'],
                                        'name' => $item['StyleName'],
                                        'status' => $item['StyleStatus'],
                                        'frame_style_data' => $jsonItemsWithSizeData,
                                        'frame_lab_data' => '',
                                        'created_at' => now()
                                    ]
                                );
                            } catch (Exception $e) {
                                // Log error or handle
                            }
                        }
                    }
                }
                return true;
            }
        } catch (Exception $e) {
            return false;
        }

        return false;
    }

    /**
     * Function to save style data
     */
    public function getFrameSize(Client $client, int $style_id): array
    {
        $url = $this->ApiBaseUrl . '/styles/' . $style_id . '/configurations?auth=' . $this->authToken;

        try {
            $result = [];
            $response = $client->request('GET', $url);
            if ($response->getStatusCode() == 200) {
                $data = json_decode($response->getBody(), true);
                foreach ($data['Configurations'] as $item) {
                    $result[] = json_encode($item);
                }
                return $result;
            }
        } catch (Exception $e) {
            return [];
        }

        return [];
    }

    public function updateNonRx(Request $request): void
    {
        // Implementation
    }

    /**
     * Function to save package data to database
     */
    public function savePackages(Request $request): JsonResponse
    {
        $params = $request->get('selectedPackages');
        $conditions = [
            'manufacturer_id' => $params['manufacturerId'],
            'brand_id' => $params['brandId'],
        ];

        if ($params['selectType'] == 'frame') {
            $conditions['model_id'] = $params['dynamicId'];
        } else {
            $conditions['configuration_fpc'] = $params['dynamicId'];
        }

        $package = FrameStyles::where($conditions)->first();

        if ($package && $params['action'] == 'save') {
            if (!empty($package->package_data)) {
                $existingData = json_decode($package->package_data, true);
                $newData = json_decode($params['packageResponse'], true);
                $updatedData = $existingData + $newData;
                $mergedJson = json_encode($updatedData);

                if ($params['selectType'] == 'frame') {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('model_id', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => json_encode($updatedData)]);
                } else {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('configuration_fpc', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => json_encode($updatedData)]);
                }
            } else {
                if ($params['selectType'] == 'frame') {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('model_id', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => $params['packageResponse'], 'tier' => 1]);
                } else {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('configuration_fpc', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => $params['packageResponse'], 'tier' => 1]);
                }
            }

            // clearCustomCache('frame_info');
            return response()->json(['status' => 'success', 'message' => 'Package assign to model', 'data' => '']);
        } elseif ($package && $params['action'] == 'unset') {
            $packageDecode = json_decode($params['packageResponse'], true);
            $packageKey = array_values($packageDecode);
            $params['packageResponse'] = $package->package_data;
            $packageDecode = json_decode($params['packageResponse'], true);
            $packageKey = array_search($packageKey[0], $packageDecode);
            $packageCount = count($packageDecode);

            if ($packageKey !== false) {
                unset($packageDecode[$packageKey]);
                $updatedPackageData = json_encode($packageDecode);
            } else {
                $updatedPackageData = $params['packageResponse'];
            }

            if ($params['selectType'] == 'frame') {
                if ($packageCount > 1) {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('model_id', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => $updatedPackageData]);
                    return response()->json(['status' => 'warning', 'message' => 'Package removed for model', 'data' => '']);
                } else {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('model_id', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => '', 'tier' => '']);
                }
            } else {
                if ($packageCount > 1) {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('configuration_fpc', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => $updatedPackageData]);
                    return response()->json(['status' => 'warning', 'message' => 'Package removed for model', 'data' => '']);
                } else {
                    FrameStyles::where('manufacturer_id', $params['manufacturerId'])
                        ->where('configuration_fpc', $params['dynamicId'])
                        ->where('brand_id', $params['brandId'])
                        ->update(['package_data' => '', 'tier' => '']);
                }
            }

            // clearCustomCache('frame_info');
        } else {
            return response()->json(['status' => 'error', 'message' => 'Package already assigned!', 'data' => '-']);
        }

        return response()->json(['status' => 'success']);
    }

    /**
     * Function to save Tier Prices
     */
    public function saveTierPrice(Request $request): JsonResponse
    {
        $params = $request->get('selectedTierPrice');
        $field = $params['type'];
        $column = $params['selectType'] == 'model' ? 'configuration_fpc' : 'model_id';
        $onlyUpdateEmpty = $params['onlyUpdateEmpty'] ?? false;

        $validFields = ['tier', 'charge', 'price', 'non_rx', 'out_edge', 'A', 'B', 'ED', 'DBL'];
        if (!in_array($field, $validFields)) {
            return response()->json(['status' => 'error', 'message' => 'Invalid field']);
        }

        try {
            $query = FrameStyles::where($column, $params['dynamicId']);
            $query->update([$field => $params['selectedVal']]);
            // clearCustomCache('frame_info');
            return response()->json(['status' => 'success', 'message' => 'Updated successfully']);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Function to get image by UPC and MID
     */
    public function getFrameImage(Request $request): JsonResponse
    {
        $frame = FrameStyles::select('image')->where('upc', $request->get('upc'))->first();
        $imagePath = 'public/assets/img/models/' . $frame->image;
        if ($frame->image) {
            $imageUrl = url($imagePath);
        } else {
            $imageUrl = $this->ApiBaseUrl . "/images/" . $request->get('manufacturer_id') . "/" . $request->get('upc') . "?auth=" . $this->authToken . "&size=Small";
        }
        return response()->json(['imageUrl' => $imageUrl]);
    }

    /**
     * Function to import in-house model through CSV
     */
    public function importModel(Request $request): JsonResponse
    {
        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '-1');

        if (!$request->hasFile('csv_file')) {
            return response()->json(['success' => false, 'message' => 'No file uploaded.'], 400);
        }

        $file = $request->file('csv_file');
        if ($file->getClientOriginalExtension() != 'csv') {
            return response()->json(['success' => false, 'message' => 'Only CSV files are allowed.'], 400);
        }

        $path = $file->getRealPath();
        $csv = Reader::createFromPath($path, 'r');
        $csv->setHeaderOffset(0);
        $headers = array_map(fn($header) => str_replace(' ', '', $header), $csv->getHeader());
        $records = $csv->getRecords();

        foreach ($records as $record) {
            $record = array_combine($headers, array_values($record));

            // Skip records missing required fields
            if (!isset($record['ManufacturerName']) || !isset($record['BrandName']) || !isset($record['ModelName'])) {
                continue;
            }

            $manufacturer = FrameManufacturer::firstOrCreate(
                ['manufacturer_name' => $record['ManufacturerName']],
                [
                    'manufacturer_id' => FrameManufacturer::max('manufacturer_id') + 1,
                    'market' => self::MARKET,
                    'status' => self::STATUS,
                ]
            );

            $brand = FrameBrands::firstOrCreate(
                ['brand_name' => $record['BrandName']],
                [
                    'brand_id' => FrameBrands::max('brand_id') + 1,
                    'manufacturer_id' => $manufacturer->manufacturer_id,
                    'market' => self::MARKET,
                    'status' => self::STATUS,
                ]
            );

            $existingFrameStyle = FrameStyles::where('name', $record['ModelName'])
                ->whereNull('collection_id')
                ->first();

            $modelId = $existingFrameStyle ? $existingFrameStyle->model_id : FrameStyles::max('model_id') + 1;

            $frameStyleData = [
                'ManufacturerName' => $manufacturer->manufacturer_name,
                'BrandName' => $brand->brand_name,
                'EdgeType' => $record['EdgeType'] ?? '',
                'FrameColor' => $record['FrameColor'] ?? '',
                'EyeSize' => $record['EyeSize'] ?? '',
                'BridgeSize' => $record['BridgeSize'] ?? '',
                'TempleLength' => $record['TempleLength'] ?? '',
                'A' => $record['A'] ?? '',
                'B' => $record['B'] ?? '',
                'ED' => $record['ED'] ?? '',
                'DBL' => $record['DBL'] ?? '',
                'UPC' => $record['UPC'] ?? '',
                'ConfigurationFPC' => $record['UPC'] ?? '',
                'SKU' => $record['UPC'] ?? '',
            ];

            FrameStyles::updateOrCreate(
                ['configuration_fpc' => $record['UPC'] ?? ''],
                [
                    'model_id' => $modelId,
                    'manufacturer_id' => $manufacturer->manufacturer_id,
                    'brand_id' => $brand->brand_id,
                    'color' => $record['FrameColor'] ?? '',
                    'size' => ($record['EyeSize'] ?? '') . '-' . ($record['BridgeSize'] ?? '') . '-' . ($record['TempleLength'] ?? ''),
                    'name' => $record['ModelName'],
                    'price' => $record['WholesalePrice'] ?? null,
                    'image' => $record['Image'] ?? null,
                    'status' => self::STATUS,
                    'frame_style_data' => json_encode($frameStyleData),
                ]
            );
        }

        return response()->json(['success' => true, 'message' => 'CSV imported successfully.']);
    }

    public function updateStyleATabled(): bool
    {
        return true;
    }
}