<?php

namespace App\Http\Controllers\Frontend\Auth;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Contracts\Auth\PasswordBroker;
use App\Http\Controllers\Controller;
use App\Models\Auth\User;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

/**
 * Class ForgotPasswordController.
 */
class ForgotPasswordController extends Controller
{
    /**
     * Display the form to request a password reset link.
     *
     * @return View
     */
    public function showLinkRequestForm(): View
    {
        return view('frontend.auth.passwords.email');
    }

    /**
     * Validate the email for the given request.
     *
     * @param Request $request
     * @return void
     */
    protected function validateEmail(Request $request): void
    {
        if (!empty($request->get('ac_code'))) {
            $request->validate(['ac_code' => 'required|exists:users,ac_code']);
        } else {
            $request->validate(['email' => 'required|email']);
        }
    }

    /**
     * Get the needed authentication credentials from the request.
     *
     * @param Request $request
     * @return array
     */
    protected function credentials(Request $request): array
    {
        if (!empty($request->get('ac_code'))) {
            return $request->only('ac_code');
        } else {
            return $request->only('email');
        }
    }

    /**
     * Get the response for a successful password reset link.
     *
     * @param Request $request
     * @param string $response
     * @return RedirectResponse
     */
    protected function sendResetLinkResponse(Request $request, string $response): RedirectResponse
    {
        return back()->with('status', trans($response))->withFlashSuccess('Reset password link has been sent it to your email address.');
    }

    /**
     * Get the response for a failed password reset link.
     *
     * @param Request $request
     * @param string $response
     * @return RedirectResponse
     */
    protected function sendStaffResetLinkFailedResponse(Request $request, string $response): RedirectResponse
    {
        return back()
            ->withInput($request->only('ac_code'))
            ->withErrors(['ac_code' => trans($response)]);
    }

    /**
     * Send a reset link to the given user.
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function sendResetLinkEmail(Request $request): RedirectResponse
    {
        $this->validateEmail($request);

        if (!empty($request->get('ac_code'))) {
            $model = User::where('ac_code', $request->get('ac_code'))->first();
            if (!empty($model) && $model->isCustomerStaff()) {
                return $this->sendStaffResetLinkFailedResponse($request, 'Please ask your system admin to reset your password.');
            }

            // Custom logic: send reset link for the user with this ac_code
            $token = Str::random(60);
            DB::table('password_resets')->updateOrInsert(
                ['email' => $model->email, 'ac_code' => $model->ac_code],
                ['token' => Hash::make($token), 'created_at' => now()]
            );
            $model->sendPasswordResetNotification($token);
            $response = Password::RESET_LINK_SENT;

            return $response == Password::RESET_LINK_SENT
                ? $this->sendResetLinkResponse($request, $response)
                : $this->sendResetLinkFailedResponse($request, $response);
        }

        // Handle password reset for email without ac_code
        $user = User::where('email', $request->email)->first();
        if (!$user) {
            return $this->sendResetLinkFailedResponse($request, 'We can\'t find a user with that email address.');
        }

        // Check if there are multiple users with this email
        $userCount = User::where('email', $request->email)->count();

        if ($userCount > 1) {
            // Multiple users with same email - require ac_code
            return back()->withErrors(['email' => 'Multiple accounts found with this email. Please provide your AC Code.']);
        }

        // Single user - proceed with standard reset
        $response = $this->broker()->sendResetLink(
            $this->credentials($request)
        );

        return $response == Password::RESET_LINK_SENT
            ? $this->sendResetLinkResponse($request, $response)
            : $this->sendResetLinkFailedResponse($request, $response);
    }

    /**
     * Get business name by ac_code via AJAX.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getBusinessNameByAcCode(Request $request): JsonResponse
    {
        $request->validate(['ac_code' => 'required|exists:users,ac_code']);

        $user = User::where('ac_code', $request->ac_code)->first();

        return response()->json([
            'business_name' => $user->business_name ?? ''
        ]);
    }

    /**
     * Get the broker to be used during password reset.
     *
     * @return PasswordBroker
     */
    public function broker(): PasswordBroker
    {
        return Password::broker();
    }

    /**
     * Get the response for a failed password reset link.
     *
     * @param Request $request
     * @param string $response
     * @return RedirectResponse
     */
    protected function sendResetLinkFailedResponse(Request $request, string $response): RedirectResponse
    {
        return back()
            ->withInput($request->only('email'))
            ->withErrors(['email' => trans($response)]);
    }
}