<?php

namespace App\Http\Controllers\Frontend\Auth;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Contracts\Auth\StatefulGuard;
use App\Helpers\Auth\AuthHelper;
use App\Exceptions\GeneralException;
use App\Http\Controllers\Controller;
use App\Helpers\Auth\SocialiteHelper;
use App\Events\Frontend\Auth\UserLoggedIn;
use App\Events\Frontend\Auth\UserLoggedOut;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

/**
 * Class LoginController.
 */
class LoginController extends Controller implements AuditableContract
{
    use Auditable;

    protected bool $auditEnabled = false;

    /**
     * Where to redirect users after login.
     *
     * @return string
     */
    public function redirectPath($user): string
    {
        return match (true) {
            $user->isAdmin() || $user->isLabManager() || $user->isLabStaff() => route('admin.orders.list'),
            $user->isCustomerAdmin() => route('frontend.user.index'),
            $user->isCustomerStaff() => route('frontend.user.common.dashboard'),
            default => route('frontend.user.index'),
        };
    }

    /**
     * @return View
     */
    public function showCustomerLoginForm(): View
    {
        return view('frontend.customers.login')
            ->withSocialiteLinks((new SocialiteHelper)->getSocialLinks());
    }

    /**
     * @return View
     */
    public function showLoginForm(): View
    {
        return view('frontend.auth.login')
            ->withSocialiteLinks((new SocialiteHelper)->getSocialLinks());
    }

    /**
     * Get the login username to be used by the controller.
     *
     * @return string
     */
    public function username(): string
    {
        $login = request()->input('ac_code');
        $field = config('access.users.customer_username');

        request()->merge([$field => $login]);

        return $field;
    }

    /**
     * Validate the user login request.
     *
     * @param Request $request
     * @return void
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function validateLogin(Request $request): void
    {
        $request->validate([
            $this->username() => 'required|string',
            'password' => 'required|string',
        ]);
    }

    /**
     * The user has been authenticated.
     *
     * @param Request $request
     * @param mixed $user
     * @param string $type
     *
     * @throws GeneralException
     * @return RedirectResponse
     */
    protected function authenticated(Request $request, $user, string $type = 'view backend'): ?RedirectResponse
    {
        // Check to see if the users account is confirmed and active
        if (!$user->isConfirmed()) {
            auth()->logout();
            // If the user is pending (account approval is on)
            if ($user->isPending()) {
                throw new GeneralException(__('exceptions.frontend.auth.confirmation.pending'));
            }

            // Otherwise see if they want to resent the confirmation e-mail

            throw new GeneralException(__('exceptions.frontend.auth.confirmation.resend', ['url' => route('frontend.auth.account.confirm.resend', e($user->{$user->getUuidName()}))]));
        }

        if (!$user->isActive()) {
            auth()->logout();

            throw new GeneralException(__('exceptions.frontend.auth.deactivated'));
        }

        if (!$user->can($type)) {
            auth()->logout();

            throw new GeneralException('You are not allowed to access this system');
        }

        event(new UserLoggedIn($user));

        if (config('access.users.single_login')) {
            auth()->logoutOtherDevices($request->password);
        }

        if (auth()->user()->can('frontend admin')) {
            return redirect()->route('frontend.user.common.dashboard');
        }

        return redirect()->intended($this->redirectPath($user));
    }

    /**
     * Log the user out of the application.
     *
     * @param Request $request
     *
     * @return RedirectResponse
     */
    public function logout(Request $request): RedirectResponse
    {
        // Remove the socialite session variable if exists
        if (app('session')->has(config('access.socialite_session_name'))) {
            app('session')->forget(config('access.socialite_session_name'));
        }

        $redirectUrl = $request->user()->can('view frontend') ? 'login' : 'frontend.auth.login';

        // Remove any session data from backend
        resolve(AuthHelper::class)->flushTempSession();

        // Fire event, Log out user, Redirect
        event(new UserLoggedOut($request->user()));

        // Laravel specific logic - temporarily disable auditing during logout
        $user = $request->user();
        if ($user && method_exists($user, 'disableAuditing')) {
            $user->disableAuditing();
        }

        $this->guard()->logout();
        $request->session()->invalidate();

        // Re-enable auditing if it was disabled
        if ($user && method_exists($user, 'enableAuditing')) {
            $user->enableAuditing();
        }

        return redirect()->route($redirectUrl);
    }

    /**
     * @return RedirectResponse
     */
    public function logoutAs(): RedirectResponse
    {
        // If for some reason route is getting hit without someone already logged in
        if (!auth()->user()) {
            return redirect()->route('frontend.auth.login');
        }

        // If admin id is set, relogin
        if (session()->has('admin_user_id') && session()->has('temp_user_id')) {
            // Save admin id
            $admin_id = session()->get('admin_user_id');

            resolve(AuthHelper::class)->flushTempSession();

            // Re-login admin
            auth()->loginUsingId((int) $admin_id);

            // Redirect to backend user page
            return redirect()->route('admin.auth.user.index');
        }

        resolve(AuthHelper::class)->flushTempSession();

        // Temporarily disable auditing during logout
        $user = auth()->user();
        if ($user && method_exists($user, 'disableAuditing')) {
            $user->disableAuditing();
        }

        auth()->logout();

        // Re-enable auditing if it was disabled
        if ($user && method_exists($user, 'enableAuditing')) {
            $user->enableAuditing();
        }

        return redirect()->route('frontend.auth.login');
    }

    /**
     * Handle a customer login request to the application.
     *
     * @param Request $request
     * @return RedirectResponse|\Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function customerLogin(Request $request): RedirectResponse
    {
        $this->validateLogin($request);

        if ($this->attemptLogin($request)) {
            return $this->sendCustomerLoginResponse($request);
        }

        return $this->sendFailedLoginResponse($request);
    }

    /**
     * Send the response after the user was authenticated.
     *
     * @param Request $request
     * @return RedirectResponse
     */
    protected function sendCustomerLoginResponse(Request $request): RedirectResponse
    {
        $request->session()->regenerate();

        $this->clearLoginAttempts($request);

        return $this->authenticated($request, $this->guard()->user(), 'view frontend')
            ?: redirect()->intended($this->redirectPath($this->guard()->user()));
    }

    /**
     * Attempt to log the user into the application.
     *
     * @param Request $request
     * @return bool
     */
    protected function attemptLogin(Request $request): bool
    {
        return $this->guard()->attempt(
            $this->credentials($request), $request->filled('remember')
        );
    }

    /**
     * Get the needed authorization credentials from the request.
     *
     * @param Request $request
     * @return array
     */
    protected function credentials(Request $request): array
    {
        return $request->only($this->username(), 'password');
    }

    /**
     * Send the response after the user was authenticated.
     *
     * @param Request $request
     * @return RedirectResponse
     */
    protected function sendLoginResponse(Request $request): RedirectResponse
    {
        $request->session()->regenerate();

        $this->clearLoginAttempts($request);

        return $this->authenticated($request, $this->guard()->user())
            ?: redirect()->intended($this->redirectPath($this->guard()->user()));
    }

    /**
     * Get the failed login response instance.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    protected function sendFailedLoginResponse(Request $request)
    {
        throw \Illuminate\Validation\ValidationException::withMessages([
            $this->username() => [trans('auth.failed')],
        ]);
    }

    /**
     * Get the guard to be used during authentication.
     *
     * @return StatefulGuard
     */
    protected function guard(): StatefulGuard
    {
        return \Illuminate\Support\Facades\Auth::guard();
    }

    /**
     * Clear the login locks for the given user credentials.
     *
     * @param Request $request
     * @return void
     */
    protected function clearLoginAttempts(Request $request): void
    {
        // Throttle logic can be implemented here if needed
    }
}