<?php

namespace App\Http\Controllers\Frontend\User;

use App\Http\Controllers\Controller;
use App\Models\Backend\UserProfile;
use App\Repositories\Frontend\Auth\UserRepository;
use App\Http\Requests\Frontend\User\UpdateProfileRequest;
use App\Models\Auth\User;
use Illuminate\Support\Facades\Hash;

/**
 * Class ProfileController.
 */
class ProfileController extends Controller
{
    /**
     * @var UserRepository
     */
    protected $userRepository;

    /**
     * ProfileController constructor.
     *
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * @param UpdateProfileRequest $request
     *
     * @throws \App\Exceptions\GeneralException
     * @return mixed
     */
    public function update(UpdateProfileRequest $request)
    {
        $message = __('strings.frontend.user.profile_updated');

        $step = $request->get('steps');

        $result = match ($step) {
            'personal_info' => $this->userRepository->update(
                $request->user(),
                $request->all()
            ),
            'avatar' => $this->userRepository->updateAvatar(
                $request->user(),
                $request->all(),
                $request->hasFile('avatar_location') ? $request->file('avatar_location') : false
            ),
            'settings' => $this->userRepository->updateSettings(
                $request->user(),
                $request->all()
            ),
            'staff_permissions' => $this->handleStaffPermissions($request),
            'update_status' => $this->handleUpdateStatus($request),
            'update_staff_password' => $this->handleUpdateStaffPassword($request),
            default => null,
        };

        if ($step === 'avatar' && $request->ajax()) {
            return response()->json([
                'message' => __('strings.frontend.user.profile_updated'),
                'image' => $request->user()->picture
            ]);
        }

        if ($result === null) {
            if ($request->ajax()) {
                return response()->json(['status' => false, 'message' => 'Profile is not updated!']);
            }
            return redirect()->route('frontend.user.account')->withFlashDanger('Profile is not updated!');
        }

        if ($request->ajax()) {
            return response()->json([
                'status' => true,
                'message' => $message,
                'image' => $request->user()->picture
            ]);
        }
        return redirect()->route('frontend.user.account')->withFlashSuccess(__('strings.frontend.user.profile_updated'));
    }

    /**
     * Handle staff permissions update.
     */
    private function handleStaffPermissions(UpdateProfileRequest $request): bool
    {
        $user = User::findOrFail($request->get('user_id'));
        $user->permission_order_placed = $request->boolean('permission_order_placed');
        $user->permission_prices_invoices = $request->boolean('permission_prices_invoices');
        $user->save();
        return true;
    }

    /**
     * Handle update status.
     */
    private function handleUpdateStatus(UpdateProfileRequest $request): bool
    {
        $user = User::findOrFail($request->get('user_id'));
        $user->active = $request->boolean('status');
        $user->save();
        return true;
    }

    /**
     * Handle update staff password.
     */
    private function handleUpdateStaffPassword(UpdateProfileRequest $request): bool
    {
        $user = User::findOrFail($request->get('user_id'));
        $user->password = Hash::make($request->get('password'));
        $user->save();
        return true;
    }

    /**
     * @param UpdateProfileRequest $request
     *
     * @throws \App\Exceptions\GeneralException
     * @return mixed
     */
    public function updateApi(UpdateProfileRequest $request)
    {
        if ($request->get('steps') === 'avatar') {
            $this->userRepository->updateAvatar(
                $request->user(),
                $request->all(),
                $request->hasFile('avatar_location') ? $request->file('avatar_location') : false
            );
            return response()->json([
                'status' => true,
                'message' => __('strings.frontend.user.profile_updated'),
                'image' => $request->user()->picture
            ]);
        }
        return response()->json([
            'status' => false,
            'message' => 'Something went wrong. Please try again!'
        ]);
    }
}