<?php

declare(strict_types=1);

namespace App\Http\Controllers\Frontend\User;

use App\Models\Auth\Role;
use App\Models\Auth\User;
use Illuminate\Http\Request;
use App\Models\Backend\UserProfile;
use App\Http\Controllers\Controller;
use App\Models\Backend\CustomerLinkStaff;
use App\Events\Backend\Auth\User\UserDeleted;
use App\Repositories\Backend\Auth\RoleRepository;
use App\Repositories\Frontend\Auth\UserRepository;
use App\Http\Requests\Frontend\User\ShowUserRequest;
use App\Http\Requests\Frontend\User\StoreUserRequest;
use App\Http\Requests\Frontend\User\UpdateUserRequest;
use App\Http\Requests\Frontend\User\ManageUserRequest;
use App\Repositories\Backend\Auth\PermissionRepository;
use App\Exceptions\GeneralException;
use App\Helpers\Auth\AuthHelper;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class UserController extends Controller
{
    protected UserRepository $userRepository;

    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    public function index(ShowUserRequest $request): View
    {
        $pageCount = (int) config('custom.page_size', 20);
        if ($request->filled('page-no')) {
            $pageCount = (int) $request->input('page-no') + $pageCount;
        }

        $users = $this->userRepository->getActivePaginated($pageCount, 'id', 'asc');

        if ($request->ajax()) {
            return view('partials.user-table')->withUsers($users)->withPageCount($pageCount);
        }

        return view('frontend.user.index')->withUsers($users)->withPageCount($pageCount);
    }

    public function loginAs(ManageUserRequest $request, User $user): RedirectResponse
    {
        if (session()->has('admin_user_id') && session()->has('temp_user_id')) {
            if ($request->user()->id === $user->id || (int) session()->get('temp_user_id') === $user->id) {
                throw new GeneralException('Do not try to login as yourself.');
            }
            resolve(AuthHelper::class)->flushTempSession();

            auth()->loginUsingId($user->id);

            return redirect()->route(home_route());
        }

        return redirect()->route(home_route());
    }

    public function create(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository): View
    {
        return view('frontend.user.create')
            ->withRoles($roleRepository->with('permissions')->get(['id', 'name']))
            ->withUserCode($this->userRepository->acCode());
    }

    public function store(StoreUserRequest $request, User $user, Role $role): RedirectResponse
    {
        $user = $this->userRepository->create($request->all());
        if ($user) {
            return redirect()->route('frontend.user.account', '#staff_member')
                ->with('flash_success', __('alerts.backend.users.created'));
        }

        throw new GeneralException(__('exceptions.backend.access.users.create_error'));
    }

    public function show(ShowUserRequest $request, User $user): View
    {
        return view('frontend.user.show')
            ->withUser($user)
            ->withProfile($user->profile);
    }

    public function edit(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository, User $user): View
    {
        return view('frontend.user.edit')
            ->withUser($user)
            ->withProfile($user->profile)
            ->withRoles($roleRepository->get())
            ->withUserRoles($user->roles->pluck('name')->all())
            ->withPermissions($permissionRepository->get(['id', 'name']))
            ->withUserPermissions($user->permissions->pluck('name')->all());
    }

    public function update(UpdateUserRequest $request, User $user): RedirectResponse
    {
        $this->userRepository->update($user, $request->all());

        return redirect()->route('frontend.user.account', '#staff_member')
            ->with('flash_success', __('alerts.backend.users.updated'));
    }

    public function destroy(ManageUserRequest $request, User $user): RedirectResponse
    {
        $this->userRepository->deleteById($user->id);
        $this->userRepository->forceDelete($user);

        event(new UserDeleted($user));

        return redirect()->route('frontend.user.account', '#staff_member')
            ->with('flash_success', __('alerts.backend.users.deleted'));
    }

    public function notifyAdmin(Request $request): JsonResponse
    {
        $changes = json_decode($request->getContent(), true)['requestData'] ?? [];
        $user = auth()->user();
        $accountCode = $user->ac_code ?? '';
        $profile = $user->profile;
        $businessName = $profile ? $profile->business_name ?? '' : '';
        $replyToEmail = $user->email ?? '';
        $htmlTable = self::generateHtmlTable($changes);
        $currentDate = now()->format('d-M-Y h:i a');
        $subject = 'Customer Profile Change Request, ' . htmlspecialchars($accountCode) . ' - ' . htmlspecialchars($businessName) . ' ' . $currentDate;
        $emailTo = 'info@lens-shapers.com';

        sendEmail($htmlTable, $emailTo, $replyToEmail, $subject);

        return response()->json([
            'success' => true,
            'message' => 'Admin has been notified successfully.',
        ]);
    }

    private function generateHtmlTable(array $changes): string
    {
        $html = '<table style="border-collapse: collapse; width: 100%;">';
        $html .= '<tr style="background-color: #f2f2f2;">';
        $html .= '<th style="border: 1px solid #ddd; padding: 8px;">Field</th>';
        $html .= '<th style="border: 1px solid #ddd; padding: 8px;">Old Value</th>';
        $html .= '<th style="border: 1px solid #ddd; padding: 8px;">New Value</th>';
        $html .= '</tr>';

        foreach ($changes as $field => $values) {
            $html .= '<tr>';
            $html .= '<td style="border: 1px solid #ddd; padding: 8px;">' . htmlspecialchars((string) $field) . '</td>';
            $html .= '<td style="border: 1px solid #ddd; padding: 8px;">' . htmlspecialchars((string) ($values['old'] ?? '')) . '</td>';
            $html .= '<td style="border: 1px solid #ddd; padding: 8px;">' . htmlspecialchars((string) ($values['new'] ?? '')) . '</td>';
            $html .= '</tr>';
        }
        $html .= '</table>';

        return $html;
    }

    public function deleteEntry(Request $request): JsonResponse
    {
        $filePath = storage_path('customer-account.txt');
        $fileContents = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $userId = auth()->user()->id;

        $updatedContents = array_filter($fileContents, function ($line) use ($userId) {
            $record = json_decode($line, true);
            return !isset($record['userId']) || $record['userId'] != $userId;
        });

        file_put_contents($filePath, implode(PHP_EOL, $updatedContents) . (count($updatedContents) > 0 ? PHP_EOL : ''));

        return response()->json(['success' => true, 'message' => 'Entry deleted successfully.']);
    }

    public function requestChange(Request $request): JsonResponse
    {
        $userData = $request->input('userData', []);
        $jsonUserData = json_encode($userData);
        $filePath = storage_path('customer-account.txt');
        $userRecord = null;

        try {
            if (($userData['requestType'] ?? '') === 'insert') {
                file_put_contents($filePath, $jsonUserData . PHP_EOL, FILE_APPEND);
                $userRecord = $userData;
            } elseif (($userData['requestType'] ?? '') === 'onload') {
                if (file_exists($filePath)) {
                    $fileContents = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                    foreach ($fileContents as $line) {
                        $record = json_decode($line, true);
                        if (isset($record['userId']) && $record['userId'] == auth()->user()->id) {
                            $userRecord = $record;
                            break;
                        }
                    }
                }
            } elseif (($userData['requestType'] ?? '') === 'update') {
                $fileContents = file_exists($filePath) ? file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) : [];
                $updated = false;
                $userId = auth()->user()->id;

                foreach ($fileContents as $index => $line) {
                    $record = json_decode($line, true);
                    if (isset($record['userId']) && $record['userId'] == $userId) {
                        $mergedData = array_merge($record, $userData);
                        $fileContents[$index] = json_encode($mergedData);
                        $updated = true;
                        $userRecord = $mergedData;
                        break;
                    }
                }

                if ($updated) {
                    file_put_contents($filePath, implode(PHP_EOL, $fileContents) . PHP_EOL);
                } else {
                    file_put_contents($filePath, $jsonUserData . PHP_EOL, FILE_APPEND);
                    $userRecord = $userData;
                }

                $this->notifyAdmin($request);
            }
        } catch (\Exception $e) {
            \Log::error('Error in requestChange: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing your request.',
            ], 500);
        }

        return response()->json([
            'success' => !empty($userData['requestData']),
            'message' => !empty($userData['requestData']) ? 'Admin has been notified successfully.' : '',
            'data' => $userRecord,
        ]);
    }
}