<?php

namespace App\Models\Auth\Traits\Attribute;

use Illuminate\Support\Facades\Hash;
use App\Models\Auth\User;

/**
 * Trait UserAttribute.
 */
trait UserAttribute
{
    /**
     * @param $password
     */
    public function setPasswordAttribute($password) : void
    {
        // If password was accidentally passed in already hashed, try not to double hash it
        if (
            (\strlen($password) === 60 && preg_match('/^\$2y\$/', $password)) ||
            (\strlen($password) === 95 && preg_match('/^\$argon2i\$/', $password))
        ) {
            $hash = $password;
        } else {
            $hash = Hash::make($password);
        }

        // Note: Password Histories are logged from the \App\Observer\User\UserObserver class
        $this->attributes['password'] = $hash;
    }

    /**
     * @return string
     */
    public function getStatusLabelAttribute()
    {
        if ($this->isActive()) {
            return "<span class='badge bg-success user-status-label'>".__('labels.general.active').'</span>';
        }

        return "<span class='badge bg-danger user-status-label'>".__('labels.general.inactive').'</span>';
    }

    /**
     * @return string
     */
    public function getTypeLabelAttribute()
    {
        $role = isset($this->getRoleNames()[0]) ? $this->getRoleNames()[0] : '';
        switch ($role) {
            case config('access.users.admin_role'):
                return "<span class='badge bg-success'>".ucwords(str_replace('_', ' ', $role)).'</span>';
                break;
            case config('access.users.lab_manager'):
                return "<span class='badge bg-primary'>".ucwords(str_replace('_', ' ', $role)).'</span>';
                break;
            case config('access.users.lab_staff'):
                return "<span class='badge bg-warning'>".ucwords(str_replace('_', ' ', $role)).'</span>';
                break;
            case config('access.users.customer_admin'):
                return "<span class='badge bg-info'>".ucwords(str_replace('_', ' ', $role)).'</span>';
                break;
            case config('access.users.customer_staff'):
                return "<span class='badge bg-danger'>".ucwords(str_replace('_', ' ', $role)).'</span>';
                break;
        }

        return '';
    }

    /**
     * @return string
     */
    public function getConfirmedLabelAttribute()
    {
        if ($this->isConfirmed()) {
            if (auth()->user()->isAcessible(['backend admin']) && $this->id != 1 && $this->id != auth()->id()) {
                return '<a href="'.route('admin.auth.user.unconfirm',$this).'"
                data-bs-toggle="tooltip"
                data-bs-placement="top"
                title="'.__('buttons.backend.access.users.unconfirm').'"
                name="confirm_item">
                <span class="badge bg-success" style="cursor:pointer">'.__('labels.general.yes').'</span></a>';
            }

            return '<span class="badge bg-success">'.__('labels.general.yes').'</span>';
        }
        if(auth()->user()->isAcessible(['backend admin']))
            return '<span class="badge bg-danger">'.__('labels.general.no').'</span>';


        return '<a href="'.route('admin.auth.user.confirm', $this).'" data-bs-toggle="tooltip" data-bs-placement="top" title="'.__('buttons.backend.access.users.confirm').'" name="confirm_item"><span class="badge bg-danger" style="cursor:pointer">'.__('labels.general.no').'</span></a>';
    }

    /**
     * @return string
     */
    public function getRolesLabelAttribute()
    {
        $roles = $this->getRoleNames()->toArray();

        if (\count($roles)) {
            return implode(', ', array_map(function ($item) {
                return ucwords($item);
            }, $roles));
        }

        return 'N/A';
    }

    /**
     * @return string
     */
    public function getPermissionsLabelAttribute()
    {
        $permissions = $this->getDirectPermissions()->toArray();

        if (\count($permissions)) {
            return implode(', ', array_map(function ($item) {
                return ucwords($item['name']);
            }, $permissions));
        }

        return 'N/A';
    }

    /**
     * @return string
     */
    public function getFullNameAttribute()
    {
        return $this->last_name
            ? $this->first_name.' '.$this->last_name
            : $this->first_name;
    }

    /**
     * @return string
     */
    public function getNameAttribute()
    {
        return $this->full_name;
    }

    /**
     * @return mixed
     */
    public function getUserPictureAttribute()
    {
        return $this->getPicture(false,false);
    }

    /**
     * @return mixed
     */
    public function getPictureAttribute()
    {
        return $this->getPicture();
    }

    /**
     * @return string
     */
    public function getSocialButtonsAttribute()
    {
        $accounts = [];

        foreach ($this->providers as $social) {
            $accounts[] = '<a href="'.route(
                'admin.auth.user.social.unlink',
                [$this, $social]
            ).'" data-bs-toggle="tooltip" data-bs-placement="top" title="'.__('buttons.backend.access.users.unlink').'" data-method="delete"><i class="fab fa-'.$social->provider.'"></i></a>';
        }

        return \count($accounts) ? implode(' ', $accounts) : __('labels.general.none');
    }

    /**
     * @return string
     */
    public function getLoginAsButtonAttribute()
    {
        // If the admin is currently NOT spoofing a user
        if (! session()->has('admin_user_id') || ! session()->has('temp_user_id')) {
            //Won't break, but don't let them "Login As" themselves
            if ($this->id !== auth()->id()) {
                return '<a href="'.route(
                    'admin.auth.user.login-as',
                    $this
                ).'" class="btn btn-sm dropdown-item"><i class="fa fa-sign-in" aria-hidden="true"></i>   '.__('buttons.backend.access.users.login_as', ['user' => e($this->full_name)]).'</a> ';
            }
        }

        return '';
    }

 /**
     * @return string
     */
    public function getClearSessionButtonAttribute()
    {
        if ($this->id !== auth()->id()) {
            return '<a
                href                        ="'.route('admin.auth.user.clear-session', $this).'"
                data-trans-button-cancel    ="'.__('buttons.general.cancel').'"
                data-trans-button-confirm   ="'.__('buttons.general.continue').'"
                data-trans-title            ="'.__('strings.backend.general.are_you_sure').'"
                class                       ="btn btn-sm dropdown-item" name="confirm_item"><i class="fa fa-clock-o" aria-hidden="true"></i>   '.__('buttons.backend.access.users.clear_session').'</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getBackToAdminButtonAttribute()
    {
        // If the admin is currently NOT spoofing a user
        if (session()->has('admin_user_id') && session()->has('temp_user_id')) {
            $user = User::where('id',(int) session()->get('admin_user_id'))->first();
            //Won't break, but don't let them "Login As" themselves
            if (!empty($user) && $user->isAdmin() && session()->has('temp_user_id') == $user->id && $user->id !== auth()->id()) {
                return '<a href="'.route(
                    'frontend.user.login-as',
                    $user
                ).'" class=""><i class="fa fa-sign-in" aria-hidden="true"></i> Back to '.$user->full_name.'</a> ';
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getShowButtonAttribute()
    {
        if(auth()->user()->isAcessible(['backend view users','backend admin'])) {
            return '<a href="'.route('admin.auth.user.show', $this).'" title="'.__('buttons.general.crud.view').'" class="btn blue btn-xs"><i class="fa fa-eye"></i></a>';
        } elseif(auth()->user()->isAcessible(['frontend admin'])) {
            return '<a href="'.route('frontend.user.show', $this).'" title="'.__('buttons.general.crud.view').'" class="btn blue btn-xs"><i class="fa fa-eye"></i></a>';
        }
        return '';
    }

    /**
     * @return string
     */
    public function getEditButtonAttribute()
    {
        if(auth()->user()->isAcessible(['backend admin'])){
            return '<a href="'.route('admin.auth.user.edit', $this).'" title="'.__('buttons.general.crud.edit').'" class="btn btn-xs btn-primary"><i class="fa fa-edit"></i></a>';
        } elseif(auth()->user()->isAcessible(['frontend admin'])) {
            return '<a href="'.route('frontend.user.edit', $this).'" title="'.__('buttons.general.crud.edit').'" class="btn btn-xs btn-primary"><i class="fa fa-edit"></i></a>';
        }
        return '';
    }

    /**
     * @return string
     */
    public function getChangePasswordButtonAttribute()
    {
        if(auth()->user()->isAcessible(['backend admin']))
            return '<a href="'.route('admin.auth.user.change-password', $this).'" class="btn btn-sm dropdown-item"><i class="fa fa-unlock-alt" aria-hidden="true"></i>   '.__('buttons.backend.access.users.change_password').'</a> ';
        elseif (auth()->user()->isAcessible(['frontend admin']))
            return '<a href="'.route('frontend.auth.user.change-password', $this).'" class="btn btn-sm dropdown-item"><i class="fa fa-unlock-alt" aria-hidden="true"></i>   '.__('buttons.backend.access.users.change_password').'</a> ';
    }

    /**
     * @return string
     */
    public function getStatusButtonAttribute()
    {
        if ($this->id != auth()->id()) {
            switch ($this->active) {
                case 0:
                    return '<a href="'.route('admin.auth.user.mark', [
                        $this,
                        1,
                    ]).'" class="btn btn-sm dropdown-item">'.__('buttons.backend.access.users.activate').'</a> ';

                case 1:
                    return '<a href="'.route('admin.auth.user.mark', [
                        $this,
                        0,
                    ]).'" class="btn btn-sm dropdown-item"><i class="fa fa-power-off" aria-hidden="true"></i>   '.__('buttons.backend.access.users.deactivate').'</a> ';

                default:
                    return '';
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getConfirmedButtonAttribute()
    {
        if (! $this->isConfirmed() && ! config('access.users.requires_approval')) {
            return '<a href="'.route('admin.auth.user.account.confirm.resend', $this).'" class="btn btn-sm dropdown-item">'.__('buttons.backend.access.users.resend_email').'</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeleteButtonAttribute()
    {
        if ($this->id != auth()->id() && $this->id != 1 && auth()->user()->isAcessible('backend admin')) {
            return '
                <a href="'.route('admin.auth.user.destroy', $this).'"
                data-method="delete"
                data-trans-button-cancel="'.__('buttons.general.cancel').'"
                data-trans-button-confirm="'.__('buttons.general.crud.delete').'"
                data-trans-title="'.__('strings.backend.general.are_you_sure').'"
                class="btn btn-sm dropdown-item"><i class="fa fa-trash-o" aria-hidden="true"></i>   '.__('buttons.general.crud.delete').'</a> ';
        } elseif ($this->id != auth()->id() && $this->id != 1 || auth()->user()->isAcessible('frontend admin')) {
            return '
                <a href="'.route('frontend.user.destroy', $this).'"
                data-method="delete"
                data-trans-button-cancel="'.__('buttons.general.cancel').'"
                data-trans-button-confirm="'.__('buttons.general.crud.delete').'"
                data-trans-title="'.__('strings.backend.general.are_you_sure').'"
                class="btn btn-sm dropdown-item"><i class="fa fa-trash-o" aria-hidden="true"></i>   '.__('buttons.general.crud.delete').'</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeletePermanentlyButtonAttribute()
    {

        if (auth()->user()->isAcessible('backend admin')) {
            return '<a
                href="'.route('admin.auth.user.delete-permanently', $this).'"
                name="confirm_item"
                class="btn btn-sm btn-danger">
                <i class="fa fa-trash"
                data-bs-toggle="tooltip"
                data-bs-placement="top"
                title="'.__('buttons.backend.access.users.delete_permanently').'">
                </i></a> ';

        } elseif (auth()->user()->isAcessible('frontend admin')) {
            return '<a
                href="'.route('frontend.user.delete-permanently', $this).'"
                name="confirm_item"
                class="btn btn-sm btn-danger">
                <i class="fa fa-trash"
                data-bs-toggle="tooltip"
                data-bs-placement="top"
                title="'.__('buttons.backend.access.users.delete_permanently').'">
                </i></a> ';
        }
    }

    /**
     * @return string
     */
    public function getRestoreButtonAttribute()
    {

        if (auth()->user()->isAcessible('backend admin')) {
            return '
                <a href="'.route('admin.auth.user.restore', $this).'"
                name="confirm_item"
                class="btn btn-sm btn-primary">
                <i class="fa fa-refresh" data-bs-toggle="tooltip" data-bs-placement="top" title="'.__('buttons.backend.access.users.restore_user').'"></i></a> ';
        } else {
            return '
            <a href="'.route('frontend.user.restore', $this).'"
            name="confirm_item"
            class="btn btn-sm btn-primary">
            <i class="fa fa-refresh" data-bs-toggle="tooltip" data-bs-placement="top" title="'.__('buttons.backend.access.users.restore_user').'"></i></a> ';
        }


    }

    /**
     * @return string
     */
    public function getAddPhoneLogAttribute()
    {
        if($this->isCustomerAdmin())
            return '<a href="#stack1" class="btn btn-sm dropdown-item sbold add_comment" data-id='.$this->id.' data-name='.$this->full_name.' da data-attr="phone" data-toggle="modal"><i class="fa fa-phone-square" aria-hidden="true"></i> Phone Logs</a>';
    }

    /**
     * @return string
     */
    public function getActionButtonsAttribute()
    {
        $buttons = '';
        if ($this->trashed()) {
            $buttons .= '
    	'.$this->restore_button.'
                 '.$this->delete_permanently_button;
        }

        $buttons .= '
            '.$this->show_button.'
            '.$this->edit_button;

        if(auth()->user()->isAcessible(['view backend'])){
            $buttons .='<div class="btn-group" role="group" aria-label="'.__('labels.backend.access.users.user_actions').'">
                    <button class="btn btn-sm btn-success dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                        '.__('labels.general.more').'
                        <i class="fa fa-angle-down"></i>
                    </button>
                    <ul class="dropdown-menu" role="menu">
                        ';
                        if(auth()->user()->isAcessible(['view backend'])){
                            $buttons .= '<li>
                                            '. $this->add_phone_log.'
                                        </li>';
                        }

                        $buttons .='<li>
                            '. $this->login_as_button.'
                        </li>
                        <li>
                            '.$this->change_password_button.'
                        </li>
                        <li>
                            '.$this->delete_button.'
                        </li>
                    </ul>
            </div>';
        } elseif (auth()->user()->isAcessible(['frontend admin'])) {
            $buttons .='<div class="btn-group" role="group" aria-label="'.__('labels.backend.access.users.user_actions').'">
                    <button class="btn btn-sm btn-success dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                        '.__('labels.general.more').'
                        <i class="fa fa-angle-down"></i>
                    </button>
                    <ul class="dropdown-menu" role="menu">
                        <li>
                            '.$this->delete_button.'
                        </li>
                        <li>
                            '.$this->change_password_button.'
                        </li>
                    </ul>
            </div>';
        }
        return $buttons;
    }
}
