<?php

namespace App\Models\Auth\Traits\Method;

use Twilio\Rest\Client;
use App\Models\Auth\Role;
use Illuminate\Http\Request;
use App\Services\TwillioFax;
use Illuminate\Support\Facades\DB;
use App\Models\Backend\UserProfile;
use App\Models\Backend\UserSettings;
use App\Models\Backend\Notification;
use App\Models\Backend\CustomerLinkStaff;
use App\Models\Backend\UserDevices;
use App\Helpers\General\PusherFactory;
use App\Models\Backend\Order;

/**
 * Trait UserMethod.
 */
trait UserMethod
{
    /**
     * defining const against column 'to' in notifications
     * table to make it self descriptive
     */
    static $user_id = 'to';

    /**
     * @return mixed
     */
    public function canChangeEmail()
    {
        return config('access.users.change_email');
    }

    /**
     * @return bool
     */
    public function canChangePassword()
    {
        return ! app('session')->has(config('access.socialite_session_name'));
    }

    /**
     * @return mixed
     */
    public function userProfile()
    {
        return $this->hasOne(UserProfile::class,'user_id','id');
    }

    /**
     * @return mixed
     */
    public function userSettings()
    {
        return $this->hasOne(UserSettings::class,'user_id','id');
    }

    /**
     * @return bool
     */
    public function isGravatar()
    {
        return $this->avatar_type == 'gravatar' ? true : false;
    }


    public function removePicture(){
        if(file_exists(storage_path('app/public/'.$this->avatar_location))){
            unlink(storage_path('app/public/'.$this->avatar_location));
            return true;
        }
        return false;
    }

    /**
     * @param bool $size
     *
     * @return bool|\Illuminate\Contracts\Routing\UrlGenerator|mixed|string
     * @throws \Illuminate\Container\EntryNotFoundException
     */
    public function getPicture($size = false,$gravatar = true)
    {
        switch ($this->avatar_type) {
            case 'storage':
                if (!empty($this->avatar_location) && file_exists(storage_path('app/public/'.$this->avatar_location))) {
                    return url('storage/'.$this->avatar_location);
                } else {
                    if($gravatar)
                        return url('public/assets/global/img/default.jpg');
                    else
                        return '';
                }
            case 'gravatar': default:
                if($gravatar)
                    return url('public/assets/global/img/default.jpg');
                else
                    return '';
        }

        $social_avatar = $this->providers()->where('provider', $this->avatar_type)->first();
        if ($social_avatar && strlen($social_avatar->avatar)) {
            return $social_avatar->avatar;
        }

        return false;
    }

    /**
     * @param $provider
     *
     * @return bool
     */
    public function hasProvider($provider)
    {
        foreach ($this->providers as $p) {
            if ($p->provider == $provider) {
                return true;
            }
        }

        return false;
    }

    /**
     * @return mixed
     */
    public function isAdmin()
    {
        return $this->hasRole(config('access.users.admin_role'));
    }

    /**
     * @return mixed
     */
    public function isLabManager() {
        return $this->hasRole(config('access.users.lab_manager'));
    }

    /**
     * @return mixed
     */
    public function isLabStaff() {
        return $this->hasRole(config('access.users.lab_staff'));
    }

    /**
     * @return mixed
     */
    public function isCustomerAdmin() {
        return $this->hasRole(config('access.users.customer_admin'));
    }

    /**
     * @return mixed
     */
    public function isCustomerStaff() {
        return $this->hasRole(config('access.users.customer_staff'));
    }

    public function isStatementAccessible()
    {
        if(!$this->isAdmin()){
            $user = $this->getCustomerAdmin();
            if(!empty($user->profile)){
                return empty($user->profile->cust_group_cust_no) || $user->profile->cust_group_cust_no == 0 ? true : false;
            }
        }
        return true;
    }

    /**
     * @return bool
     */
    public function isActive()
    {
        return $this->active;
    }

    /**
     * @return bool
     */
    public function isConfirmed()
    {
        return $this->confirmed;
    }

    /**
     * @return bool
     */
    public function isTempPass(){
        return $this->is_temp_pass;
    }

    /**
     * @return bool
     */
    public function isPending()
    {
        return config('access.users.requires_approval') && ! $this->confirmed;
    }

    /**
     * @return bool
     */
    public function isAcessible($permission)
    {
        return $this->hasAnyPermission($permission);
    }

    /**
     * @return bool
     */
    public function isBackendUser()
    {
        return $this->hasAnyPermission(['view backend']);
    }

    /**
     * @return bool
     */
    public function isFrontendUser()
    {
        return $this->hasAnyPermission(['view frontend']);
    }

    /**
     * @return string
     */
    public function getBusinessName()
    {
        if(!$this->isAdmin()){
            $user = $this->getCustomerAdmin();
            if(!empty($user->profile))
                return $user->profile->business_name;
        }
        return '';
    }

    /**
     * @return bool
     */
    public function isOrderPlacedPermission()
    {
        if($this->isCustomerStaff()){
            if($this->permission_order_placed){
                return true;
            }
            return false;
        }
        return true;
    }

    /**
     * @return bool
     */
    public function isOrderPricesInvoicesPermission()
    {
        if($this->isCustomerStaff()){
            if($this->permission_prices_invoices){
                return true;
            }
            return false;
        }
        return true;
    }

    /**
     * @return bool
     */
    public function isStockUser()
    {
        $userProfile = $this->userProfile;
        if($this->isCustomerStaff()){
            $userProfile = $this->getCustomerAdmin()->userProfile;
        }
        if(!empty($userProfile) && $userProfile->cust_prlist_stock == 720)
            return true;

        return false;
    }

    /**
     * @param   $user
     *
     * @return  model
     */

    public function updateProfile($data){
        if(!$this->isCustomerAdmin())
            return false;

        if(isset($data['religion']) && !empty($data['religion'])){
            $data['religion'] = json_encode($data['religion']);
        }

        $modelProfile = UserProfile::where('user_id',$this->id)->first();
        if(!empty($modelProfile))
            $modelProfile->update($data);
        else{
            $data['user_id'] = $this->id;
            UserProfile::create($data);
        }

        return true;
    }

    /**
     * @return mixed
     */
    public function customerAdminLinkStaff()
    {
        return $this->hasMany(CustomerLinkStaff::class,'customer_id','id');
    }

    /**
     * @return mixed
     */
    public function customerStaffLinkAdmin()
    {
        return $this->hasOne(CustomerLinkStaff::class,'user_id','id');
    }

    /**
     * @return model
     */
    public function getCustomerStaff(){
        $assignedUsers = collect($this->customerAdminLinkStaff)->map(function ($model) {
            return $model->user_id;
        })->toArray();
        return $assignedUsers;
    }

    /**
     * @return string
     */
    public function fullName() {
        return $this->first_name . ' ' . $this->last_name;
    }

    /**
     * @return string
     */
    public function role() {
        $role = Role::find(request()->roles);
        return $role->name;
    }


    public function stats(){
        $response = [];
        $users = $this->getUserCountByRoles();
        foreach ($users as $key => $value) {
            $response[] = [
                'title'         =>  $value->getFullNameAttribute(),
                'count'         =>  $value->users_count,
                'percentage'    =>  100,
                'icon'          =>  'icon-briefcase',
                'color'         =>  self::getDashboardColorList($key),
                'order'         =>  $key,
                'url'           =>  'javascript:void(0)'
            ];
        }

        usort($response, function($a, $b) {
            return $a['order'] <=> $b['order'];
        });
        return $response;
    }

    public static function getDashboardColorList($key = -1){
        $list = [
            'blue',
            'red',
            'green',
            'purple',
            'yellow'
        ];
        if(isset($list[$key]) && !empty($list[$key]))
            return $list[$key];

        return $list;
    }

    public function getCustomerAdmin(){
        if($this->isCustomerStaff() && !empty($this->customerStaffLinkAdmin)){
            return (!empty($this->customerStaffLinkAdmin->customerAdmin)) ? $this->customerStaffLinkAdmin->customerAdmin : $this;
        }
        return $this;
    }

    public static function getUserByRoleName($name){
        return self::join('model_has_roles', 'model_has_roles.model_id', '=', 'users.id')
        ->join('roles', 'roles.id', '=', 'model_has_roles.role_id') // Join roles table
        ->where('roles.name', $name)
        ->select('users.*') // Fetch role names
        ->get();
    }

    public function getUsersList(){
        $roleIds = [];
        if($this->isBackendUser()){

            if($this->isAdmin()){
                $modelRole = Role::
                // where('name','!=',config('access.users.customer_staff'))->
                get();
            }else{
                $modelRole = Role::where('name',config('access.users.customer_admin'))->get();
            }
            $roleIds = collect($modelRole)->map(function ($model) {
                return $model->id;
            })->toArray();
        }elseif($this->isCustomerAdmin()){
            $modelRole = Role::where('name',config('access.users.customer_staff'))->get();
            $roleIds = collect($modelRole)->map(function ($model) {
                return $model->id;
            })->toArray();
        }
        $modelUsers = self::join('model_has_roles', 'model_has_roles.model_id', '=', 'users.id')
                    // ->where('active','=',self::STATUS_ACTIVE)
                    ->whereIN('role_id',$roleIds)
                    ->get();

        $modelUsers = collect($modelUsers)->map(function ($model) {
                return $model->id;
            })->toArray();
        return $modelUsers;
    }

    public function getUserCountByRoles(){
        $roles = [];
        if($this->isBackendUser()){
            $roles = Role::withCount('users')->get();
        }elseif($this->isCustomerAdmin()){
            $roles = Role::withCount('users')
                        ->where('name',config('access.users.customer_staff'))
                        ->get();
            $roles[0]->users_count = count($this->customerAdminLinkStaff);
        }
        return $roles;
    }

    /**
     * Get all the notifictions associated with a user
     *
     * @return $notifications
     */
    public function notifications() {
        if($this->isCustomerStaff() || $this->isCustomerAdmin()){
            return Notification::where('to',$this->getCustomerAdmin()->id)
            ->orderBy('created_at', 'desc')
            ->get();
        }
        return Notification::where('to',1)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Get all the unread notifictions associated with a user
     *
     * @return $notifications
     */
    public function unreadNotifications() {
        if($this->isCustomerStaff() || $this->isCustomerAdmin()){
            return Notification::where(self::$user_id,$this->getCustomerAdmin()->id)
            ->where('has_read', 0)
            ->get();
        }
        return Notification::where(self::$user_id,1)
            ->where('has_read', 0)
            ->get();
    }

    public function getSuperAdminId(){
        return 1;
    }

    /**
     * Get all the unread notifictions associated with a user
     *
     * @return $notifications
     */
    public function setAllReadNotifications() {
        $user = $this;
        if($this->isCustomerStaff() || $this->isCustomerAdmin()){
            return Notification::where(function ($query) use ($user) {
                    $query->where('to', $user->getCustomerAdmin()->id)
                          ->orWhere('to', $user->id);
                })
                ->where('has_read', 0)
                ->update(['has_read'=>1]);
        }
        return Notification::where(function ($query) use ($user) {
                    $query->where('to', 1)
                          ->orWhere('to', $user->id);
                })
                ->where('has_read', 0)
                ->update(['has_read'=>1]);
    }

    /**
     * Get users by role type
     *
     * @param $roleType
     *
     * @return $users
     */
    public static function getUserByRole($roleType) {
        $role = Role::findByType($roleType);
        if(!empty($role)){
            $user = self::join('model_has_roles', 'model_has_roles.model_id', '=', 'users.id')
                    ->where('role_id','=',$role->id)
                    ->first();

            return $user;
        }
        return null;
    }

    /**
     * Get users by role type
     *
     * @param $roleType
     *
     * @return $users
     */
    public static function getUsersByRole($roleType) {
        $role = Role::findByType($roleType);
        if(!empty($role)){
            $users = self::join('model_has_roles', 'model_has_roles.model_id', '=', 'users.id')
                    ->where('active','=',self::STATUS_ACTIVE)
                    ->where('role_id','=',$role->id)
                    ->get();

            return $users;
        }
    }

    /**
     * Send fax to user
     */
    public function sendFax($message) {

        $sid = config('access.api.twillio_fax.sid');
        $token = config('access.api.twillio_fax.token');


        $client = new Client($sid, $token);

        $client->messages->create(
            // the number you'd like to send the message to
            '+15558675309',
            [
                // A Twilio phone number you purchased at twilio.com/console
                'from' => '+15017250604',
                // the body of the text message you'd like to send
                'body' => $message
            ]
        );
    }

    /**
     * Get Address
     *
     * @return string
     */
    public function getAddress(){
        $response = '';
        if(!empty($this->userProfile)){
            if(!empty($this->userProfile->address1))
                $response .= $this->userProfile->address1.', ';
            if(!empty($this->userProfile->address2))
                $response .= $this->userProfile->address2.', ';
            if(!empty($this->userProfile->city))
                $response .= $this->userProfile->city.', ';
            if(!empty($this->userProfile->state))
                $response .= $this->userProfile->state.', ';
            if(!empty($this->userProfile->zip_code))
                $response .= $this->userProfile->zip_code;
        }
        return trim($response);
    }


    /**
     * Save Multiple Devices
     *
     * @return null
     */
    public function updateDevices($token){
        $modal = UserDevices::where('device_id',$token)->where('user_id',$this->id)->first();
        if(empty($modal)){
            UserDevices::create([
                'user_id'   =>  $this->id,
                'device_id' =>  $token
            ]);
        }else{
            $modal->updated_at = date('Y-m-d H:i:s');
            $modal->update();
        }
    }
    public function getChannelId(){
        if($this->isAdmin())
            return 'channel-admin';
        else
            return 'channel-user-'.$this->id;
    }

    /**
     * Send Web notifications/events on Multiple Devices
     *
     * @return null
     */
    public function sendWebNotifications($payloads,$channel = 'send'){
        PusherFactory::make()->trigger($this->getChannelId(), $channel, ['data' => $payloads]);
        // $url = 'https://fcm.googleapis.com/fcm/send';
        // $FcmToken = UserDevices::where('user_id',$this->id)->pluck('device_id')->all();

        // $serverKey = env('FIREBASE_SERVER_KEY');

        // $data = [
        //     "registration_ids" => $FcmToken,
        //     "notification" => [
        //         "title" => $payloads['title'],
        //         "body" => $payloads['body']
        //     ],
        //     'data'  =>  $payloads['data']
        // ];
        // $encodedData = json_encode($data);

        // $headers = [
        //     'Authorization:key=' . $serverKey,
        //     'Content-Type: application/json',
        // ];

        // $ch = curl_init();

        // curl_setopt($ch, CURLOPT_URL, $url);
        // curl_setopt($ch, CURLOPT_POST, true);
        // curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        // curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        // curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        // curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
        // // Disabling SSL Certificate support temporarly
        // curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        // curl_setopt($ch, CURLOPT_POSTFIELDS, $encodedData);
        // // Execute post
        // $result = curl_exec($ch);
        // if ($result === FALSE) {
        //     die('Curl failed: ' . curl_error($ch));
        // }
        // // Close connection
        // curl_close($ch);
    }

    public function sendWelcomeEmail($password = '')
    {
        if(empty($password)){
            $password = \Illuminate\Support\Str::random(10);
        }
        $profile = $this->profile;
        $this->password = $password;
        $this->is_temp_pass = 1;
        $this->save();
        $data = [
            'accountNumber' => $this->ac_code,
            'email' => $this->email,
            'temporaryPassword' => $password,
            'businessName' => $this->getBusinessName(),
            'firstName' => $this->first_name,
            'lastName' => $this->last_name,
            'address1' => $profile->address1,
            'address2' => $profile->address2,
            'city' => $profile->city,
            'province' => $profile->state,
            'postalCode' => $profile->zip_code,
            'country' => $profile->country,
            'phone' => $profile->phone,
            'fax' => $profile->fax, // or provide the fax number if available
        ];

        \Mail::to($this->email)->send(new \App\Mail\NewUserMail($data));
        // \Mail::to('usama485@gmail.com')->send(new \App\Mail\NewUserMail($data));
        // \Mail::to('kashif.izhar@gmail.com')->send(new \App\Mail\NewUserMail($data));

        return true;
    }

    public function getMultiPlairList(){
        $orders = Order::where('user_id', $this->getCustomerAdmin()->id)
                            ->where('created_at', '>=', \Carbon\Carbon::now()->subDays(300))
                            ->orderBy('created_at', 'desc')
                            ->get();
        $multiPlairList = [];
        foreach ($orders as $key => $value) {
            $multiPlairList[] = $value->getOrderNumber().', '.$value->invoice_no.', '.$value->patient;
        }
        return $multiPlairList;
    }

    public function getOrdersData(){
        $orders = Order::where('user_id', $this->getCustomerAdmin()->id)
                            ->where('online', 1)
                            ->where('type', Order::ORDER_RX)
                            ->orderBy('created_at', 'desc')
                            ->get();
        return $orders;
    }

    public function getStickyMessage()
    {
        $msg = '';
        $user = $this->getCustomerAdmin();
        if(!empty($user->profile)){
            $rating = strtolower($user->profile->cust_cr_rating);
            if ($rating === 'sh') {
                $msg = 'Shipping on Hold, Jobs Processing Normally';
            } elseif ($rating === 'h') {
                $msg = 'Shipping on Hold, Job Processing on Hold';
            } elseif ($rating === 'ogsh') {
                $msg = 'Buying Group Hold, Shipping on Hold, Jobs Processing Normally';
            } elseif ($rating === 'ogh') {
                $msg = 'Buying Group Hold, Shipping on Hold, Job Processing on Hold';
            }
        }

        return $msg;
    }
}
