<?php

namespace App\Models\Backend;
use Illuminate\Database\Eloquent\Model;
use App\Models\Backend\Order;
use App\Models\Backend\RxValidations;

class RxOrders extends Model
{
    public $timestamps = false;

    protected $table = "rx_orders";

    protected $guarded = [];

    /**
     * @return mixed
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    public static function getFilePath($file){
        if(file_exists(public_path('assets/img/rx-colors/'.$file)))
            return url('public/assets/img/rx-colors/'.$file);

        return false;
    }

    public static function getAllImages(){
        $response = [];
        $mediaPath = public_path('assets/img/rx-colors/');
        $files = \File::allFiles($mediaPath);
        foreach ($files as $key => $value) {
            $file = pathinfo($value);
            $path = $file['dirname'].'/'.$file['basename'];
            if(file_exists($path)){
                $response[] = [
                    'name'  =>  $file['basename'],
                    'size'  =>  \File::size($path),
                    'path'  =>  self::getFilePath($file['basename'])
                ];
            }
        }
        return $response;
    }

    public static function removeAssetsImage($fileName){
        if(file_exists(public_path('assets/img/rx-colors/'.$fileName))){
            unlink(public_path('assets/img/rx-colors/'.$fileName));
            return true;
        }
        return false;
    }

    public static function uploadAssetsImage($image){
        $response = [];
        $filename = $image->getClientOriginalName();
        if($image->move(public_path('assets/img/rx-colors'), $filename)){
            $url = url('public/assets/img/rx-colors/'.$filename);
            \Log::info('Uploaded image: ' . $filename . ' to URL: ' . $url);
            $response[] = $url;
        } else {
            \Log::error('Failed to move uploaded image: ' . $filename);
        }
        return $response;
    }

    public static function newRxOrder($order, $request) {
        if(!empty($request->data['order_id'])){
            $order = Order::where('user_id',auth()->user()->getCustomerAdmin()->id)
                            ->where('id',$request->data['order_id'])
                            ->first();
            $rxOrder = null;
            if(!empty($order)){
                $rxOrder = self::where('order_id',$order->id)->first();
                $rxOrder->update(
                    [
                        'first_name'  => $request->data['first_name'],
                        'last_name'  => $request->data['last_name'],
                        'data'  => json_encode($request->data),
                    ]
                );
            }
            return $rxOrder;
        }else{
            return parent::create([
                'order_id' => $order->id,
                'first_name'  => $request->data['first_name'],
                'last_name'  => $request->data['last_name'],
                'data'  => json_encode($request->data),
            ]);
        }
    }

    public static function getCombinationArray(){
        return [
            'lens_type' =>  [
                'sv' => [
                    'label' => 'SV',
                    'options' => [
                        'sv_distance'                   =>  'Distance',
                        'sv_intermediate_or_computer'   =>  'Intermediate / Computer',
                        'sv_reading'                    =>  'Reading'
                    ]
                ],
                'multi_focal'   =>  [
                    'label'     =>  'Multi-Focal',
                    'options'   =>  []
                ],
                'progressive'  =>  [
                    'label'     =>  'Progressive',
                    'options'   =>  [
                        'progressive_hd_digital'    =>  'HD Digital',
                        'progressive_full_custom_fit'   =>  'Full Custom Fit',
                        'progressive_intermediate_computer_reading' =>  'Intermediate (Computer + Reading)',
                        'progressive_conventional'  =>  'Conventional'
                    ]
                ],
                // 'intermediate_progressive_hd_digital'            =>  [
                //             'label' =>  'Intermediate Progressive (HD Digital)',
                //             'options'   =>  []
                // ],
                // 'custom_fit_progressive_hd_digital'            =>  [
                //             'label' =>  'Custom Fit Progressive (HD Digital)',
                //             'options'   =>  []
                // ],
                // 'intermediate_progressive_conventional'            =>  [
                //             'label' =>  'Intermediate Progressive (Conventional)',
                //             'options'   =>  []
                // ]
            ],
            'rx'    =>  [
                'distance'  =>  [
                    'dist_type'    =>  ['label'=>'','display'=>true,'type'=>'dropdown','values'=>['Order'=>'Order','Balance'=>'Balance','Do not order'=>'Do not order'],'round'=>'','symbol'=>'','class'=>'col-sm p-2 dist-type-ddl','single'=>false,'data-name'=>'dist_type','name_base'=>'data[rx][distance]','increment'=>0.25,'required_if'=>'','allow-clear'=>false],
                    'sph'         =>    ['label'=>'Sphere','step'=>'0.25','type'=>'dropdown','values'=>[''=>''],'class'=>'col-sm p-2 crawl-cs-ddl','round'=>'2','symbol'=>'','indicator'=>true,'single'=>false,'data-name'=>'sph','name_base'=>'data[rx][distance]','increment'=>0.25,'inputClass' =>  'cal-base_curve is-required','required_if'=>'','allow-clear'=>false],
                    'cyl'         =>    ['label'=>'Cylinder','step'=>'0.25','type'=>'dropdown','values'=>[''=>''],'pad'=>'','class'=>'col-sm p-2 crawl-cs-ddl','round'=>'2','symbol'=>'','indicator'=>true,'single'=>false,'data-name'=>'cyl','name_base'=>'data[rx][distance]','increment'=>0.25,'inputClass' =>  'cal-base_curve is-required','required_if'=>'rl_axis','allow-clear'=>false],
                    'axis'        =>    ['label'=>'Axis','step'=>'1','pad'=>'3','type'=>'dropdown','values'=>[''=>''],'class'=>'col-sm p-2 crawl-cs-ddl','round'=>'0','symbol'=>'°','indicator'=>false,'single'=>false,'data-name'=>'axis','name_base'=>'data[rx][distance]','increment'=>1,'inputClass' =>  'is-required','required_if'=>'rl_cyl','allow-clear'=>false],
                    'add'   =>  ['label'=>'Reading Add','display'=>true,'step'=>'0.25','type'=>'dropdown','values'=>[''=>''],'pad'=>'','value'=>'','max'=>'','round'=>'2','symbol'=>'','class'=>'col-sm p-2 crawl-cs-ddl','indicator'=>true,'single'=>false,'data-name'=>'add','name_base'=>'data[rx][add_for_reading]','increment'=>0.25,'inputClass' =>  'is-required sync-lr-field','required_if'=>'','allow-clear'=>false],
                    'strength'    =>  ['label'=>'Power Boost','display'=>true,'type'=>'dropdown','values'=>[],'round'=>'','symbol'=>'','class'=>'col-sm crawl-ddl p-0 pl-4','single'=>true,'data-name'=>'strength','name_base'=>'data[rx][distance]','increment'=>0.25,'inputClass' =>  'is-required','required_if'=>'','allow-clear'=>false],
                    'intermediate_strength'    =>  ['label'=>'Intermediate Strength','display'=>true,'type'=>'dropdown','values'=>['30'=>'30%','40'=>'40%','50'=>'50% (Default)','60'=>'60%','70'=>'70%'],'round'=>'','symbol'=>'','class'=>'col-sm p-0 pl-4','single'=>true,'data-name'=>'intermediate_strength','name_base'=>'data[rx][add_for_reading]','increment'=>0.25,'inputClass' =>  'is-required','required_if'=>'','allow-clear'=>false],
                    // 'base_curve'   =>  ['label'=>'Base Curve','display'=>true,'type'=>'dropdown','values'=>[],'round'=>'','symbol'=>'','class'=>'col-sm','single'=>true,'data-name'=>'rx_base_curve','name_base'=>'data[rx][extended_info]','increment'=>0.25],
                    // 'oc'          =>    ['label'=>'OC Height','step'=>'0.5','type'=>'number','pad'=>'','class'=>'col-md-3 col-xs-6','round'=>'1','symbol'=>'mm','indicator'=>false],
                    // 'final_rx'          =>    ['label'=>'Final Rx','step'=>'1','type'=>'label','pad'=>'','class'=>'col-md-1']
                ],
                'prism' =>  [
                    'prism1'    =>  ['label'=>'Prism 1','step'=>'0.25','type'=>'dropdown','pad'=>'','round'=>'2','symbol'=>'△','values'=>[],'data-name'=>'prism1','class'=>'crawl-cs-ddl','increment'=>0.25,'inputClass' =>  'cal-base_curve is-required','empty_check'=>'base1','required_if'=>'rl_base1'],
                    'base1' =>  ['label'=>'Base 1','type'=>'dropdown','values'=>self::getBaseArray(),'data-name'=>'base1','class'=>'','round'=>'','increment'=>0.25,'inputClass' =>  'is-required','empty_check'=>'prism1','required_if'=>'rl_prism1'],
                    'prism2'    =>  ['label'=>'Prism 2','step'=>'0.25','type'=>'dropdown','pad'=>'','round'=>'2','symbol'=>'△','values'=>[],'data-name'=>'prism2','class'=>'crawl-cs-ddl','increment'=>0.25,'inputClass' =>  'cal-base_curve is-required','empty_check'=>'base2','required_if'=>'rl_base2'],
                    'base2' =>  ['label'=>'Base 2','type'=>'dropdown','values'=>self::getBaseArray(),'data-name'=>'base2','class'=>'','round'=>'','increment'=>0.25,'inputClass' =>  'is-required','empty_check'=>'prism2','required_if'=>'rl_prism2'],
                ],
                'add_for_reading' =>  [
                    // 'sh'    =>  ['label'=>'Seg Height','display'=>true,'step'=>'0.5','type'=>'number','pad'=>'','value'=>'','max'=>'','round'=>'1','symbol'=>'mm'],
                ],
                'final_rx' =>  [
                    'sph'         =>    ['label'=>'Sphere','class'=>'col-md-4 col-xs-6','symbol'=>'','pad'=>''],
                    'cyl'         =>    ['label'=>'Cylinder','class'=>'col-md-4 col-xs-6','symbol'=>'','pad'=>''],
                    'axis'        =>    ['label'=>'Axis°','class'=>'col-md-4 col-xs-6','symbol'=>'°','pad'=>3],
                ],
                'extended_info' =>  [
                    'pd'        =>  ['label'=>'Distance PD','display'=>true,'step'=>'0.5','type'=>'dropdown','values'=>[''=>''],'class'=>'col-sm p-2 crawl-cs-ddl','ddl_class'=>'select2_fixed is-required sync-lr-field','pad'=>'','round'=>'1','symbol'=>'mm','data-name'=>'dpd','name_base'=>'data[rx][extended_info]' ,'increment'=>0.5,'single'=>false,'allow-clear'=>"false"],
                    'intermediate_pd'    =>  ['label'=>'Intermediate PD','display'=>true,'step'=>'0.5','type'=>'dropdown','values'=>[''=>''],'class'=>'col-sm p-2 crawl-cs-ddl','ddl_class'=>'select2_fixed is-required sync-lr-field','pad'=>'','round'=>'1','symbol'=>'mm','data-name'=>'ipd','name_base'=>'data[rx][extended_info]' ,'increment'=>0.5,'single'=>false,'allow-clear'=>"false"],
                    'npd'            =>  ['label'=>'Near PD','display'=>true,'step'=>'0.5','type'=>'dropdown','values'=>[''=>''],'class'=>'col-sm p-2 crawl-cs-ddl','ddl_class'=>'select2_fixed is-required sync-lr-field','pad'=>'','round'=>'1','symbol'=>'mm','data-name'=>'npd','name_base'=>'data[rx][extended_info]','increment'=>0.5,'single'=>false,'allow-clear'=>"true"],
                    'oc'    =>  ['label'=>'Fitting Height','display'=>true,'step'=>'0.5','type'=>'dropdown','values'=>[''=>''],'class'=>'col-sm p-2 crawl-cs-ddl sync-lr-field oc_field-container','ddl_class'=>'select2_fixed is-required sync-lr-field','pad'=>'','value'=>'','max'=>'','round'=>'1','symbol'=>'mm','data-name'=>'oc','name_base'=>'data[rx][extended_info]','increment'=>0.5,'single'=>false,'button'=>true,'button-label'=>'Fitting Height','button-class'=>'oc','allow-clear'=>"false"],
                    'boc'    =>  ['label'=>'BOC Height','display'=>true,'step'=>'0.5','type'=>'dropdown','values'=>[''=>''],'class'=>'col-sm p-2 crawl-cs-ddl boc_field-container','ddl_class'=>'select2_fixed sync-lr-field','pad'=>'','value'=>'','max'=>'','round'=>'1','symbol'=>'','data-name'=>'boc','name_base'=>'data[rx][add_for_reading]','increment'=>0.5,'single'=>false,'button'=>true,'button-label'=>'BOC','button-class'=>'boc','allow-clear'=>"false"],
                    'corridor'    =>  ['label'=>'Corridor','display'=>true,'type'=>'dropdown','values'=>[],'round'=>'','symbol'=>'','class'=>'col-sm p-2 crawl-cs-ddl','ddl_class'=>'select2_tags select2_fixed is-required','single'=>true,'data-name'=>'corridor','name_base'=>'data[rx][extended_info]','increment'=>0.5,'allow-clear'=>"false"],
                    // 'base_curve'         =>  ['label'=>'Base Curve','display'=>false,'step'=>'1','type'=>'number','pad'=>'','round'=>'','symbol'=>''],
                ],
                'use_shorter_corridor'   =>  ['label'=>'Use Shorter Corridor'],
                'special_info' =>  [
                    'panto'   =>  ['label'=>'Panto','pad'=>'3','type'=>'number','max'=>'999','symbol'=>'°','min'=>'0','data-name'=>'panto','type'=>'dropdown','values'=>[],'round'=>'0','increment'=>'1'],
                    'wrap'    =>  ['label'=>'Wrap','pad'=>'3','type'=>'number','max'=>'999','symbol'=>'°','min'=>'0','data-name'=>'wrap','type'=>'dropdown','values'=>[],'round'=>'0','increment'=>'1'],
                    'vertex'    =>  ['label'=>'Vertex','pad'=>'3','type'=>'number','max'=>'999','symbol'=>'°','min'=>'0','data-name'=>'vertex','type'=>'dropdown','values'=>[],'round'=>'0','increment'=>'1']
                ],
            ],
            'frame' =>  [
                'job_type'  =>  [
                    'uncut' =>  [
                        'label' =>   'Uncut'
                    ],
                    'edged_frame_to_follow' =>  [
                        'label' =>   'Edged (Frame to Follow)'
                    ],
                    'edged_no_frame_to_follow' =>  [
                        'label' =>   'Edged from Shape<br>(No frame to follow)'
                    ],
                    'lab_supplies_frame' =>  [
                        'label' =>   'Lab Supplies Frame'
                    ]
                ],
                'frame_info'    =>  [
                    'brand' =>  ['label'=>'Brand','type'=>'dropdown','inputClass'=>'is-required','data-filter'=>'model','class'=>'col-sm p-2 crawl-ddl','ddl_class'=>'select2_fixed is-required filter-frame-info','allow-clear'=>"true","data-tags"=>"true","values"=>[]],
                    'model' =>  ['label'=>'Model','type'=>'dropdown','inputClass'=>'is-required','data-filter'=>'colour','class'=>'col-sm p-2 crawl-ddl','ddl_class'=>'select2_fixed is-required filter-frame-info','allow-clear'=>"true","data-tags"=>"true","values"=>[]],
                    'colour' =>  ['label'=>'Colour','type'=>'dropdown','inputClass'=>'is-required','data-filter'=>'size','class'=>'col-sm p-2 crawl-ddl','ddl_class'=>'select2_fixed is-required filter-frame-info','allow-clear'=>"true","data-tags"=>"true","values"=>[]],
                    'size' =>  ['label'=>'Size','type'=>'dropdown','inputClass'=>'is-required','data-filter'=>'','class'=>'col-sm p-2 crawl-ddl','ddl_class'=>'select2_fixed is-required filter-frame-info','allow-clear'=>"true","data-tags"=>"true","values"=>[]],
                    'pof'   =>  ['label'=>'POF','type'=>'checkbox','help_text'=>"Patient own frames will not be adjusted and are processed at customer's risk",'inputClass'=>'','data-filter'=>'']
                ],
                'frame_type'  =>  [
                    'full_metal' =>  [
                        'label' =>   'Full Metal'
                    ],
                    'full_plastic' =>  [
                        'label' =>   'Full Plastic'
                    ],
                    'nyloon_groove' =>  [
                        'label' =>   'Nylon Groove'
                    ],
                    'rimless_drill' =>  [
                        'label' =>   'Rimless (Drill / Notch)'
                    ],
                    'metal_groove' =>  [
                        'label' =>   'Metal Groove'
                    ],
                    // 'other' =>  [
                    //     'label' =>   'Other'
                    // ]
                ],
                'frame_size'  =>  [
                    'a' =>  ['label'=>'A','step'=>'0.5','type'=>'dropdown','pad'=>'','round'=>'1','symbol'=>'mm','data-name'=>'a','class'=>'col-sm p-2 crawl-cs-ddl frame_size-fields','values'=>[],'default'=>'','increment'=>0.5,'inputClass' =>  'cal-base_curve is-required'],
                    'b' =>  ['label'=>'B','step'=>'0.5','type'=>'dropdown','pad'=>'','round'=>'1','symbol'=>'mm','data-name'=>'b','values'=>[],'class'=>'col-sm p-2 crawl-cs-ddl frame_size-fields','default'=>'','increment'=>0.5,'inputClass' =>  'is-required'],
                    // 'ed_or_shape' =>  [
                    //     'label'=>'',
                    //     'type'=>'dropdown',
                    //     'default'   =>  '',
                    //     'values'=>['shape'=>"Shape",'ed'=>'ED'],'data-name'=>'ed_or_shape','class'=>'col-sm p-2','round'=>'1'],
                    'ed'    => ['label'=>'ED','step'=>'0.5','type'=>'dropdown','pad'=>'','round'=>'1','symbol'=>'mm','data-name'=>'ed','class'=>'col-sm p-2 crawl-cs-ddl frame_size-fields','default'=>'','values'=>[''=>''],'increment'=>0.5,'inputClass' =>  'cal-base_curve is-required'],
                    'shape' =>  [
                        'label'     =>  'Shape',
                        'type'      =>  'segment',
                        'data-name'=>'shape',
                        'values'    =>  [
                            'a' =>  ['label'=>'A','img'=>'1.svg','factor'=>1.00],
                            'b' =>  ['label'=>'B','img'=>'2.svg','factor'=>1.07],
                            'c' =>  ['label'=>'C','img'=>'3.svg','factor'=>1.00],
                            'd' =>  ['label'=>'D','img'=>'4.svg','factor'=>1.065],
                            'e' =>  ['label'=>'E','img'=>'5.svg','factor'=>1.071],
                            'f' =>  ['label'=>'F','img'=>'6.svg','factor'=>1.057],
                            'g' =>  ['label'=>'G','img'=>'7.svg','factor'=>1.00],
                            'h' =>  ['label'=>'H','img'=>'8.svg','factor'=>1.00]
                        ],
                        'class'=>'col-sm p-2 frame_size-fields'
                    ],
                    'dbl'    => ['label'=>'DBL','step'=>'0.5','type'=>'dropdown','pad'=>'','round'=>'1','symbol'=>'mm','data-name'=>'dbl','class'=>'col-sm p-2 crawl-cs-ddl frame_size-fields','values'=>[''=>''],'default'=>'','increment'=>0.5,'inputClass' =>  'is-required']
                ],
                'modify_shape'  =>  [
                    'ms_a' =>  [
                        'label' =>  'A',
                        'values'    =>  [
                            'a_shape_options'  =>  ['label' =>  '','type'=>'dropdown','values'=>['Increase By:'=>'Increase By:','Decrease By:'=>'Decrease By:','Make:'=>'Make:'],'class'=>'col-md-6 col-xs-6','data-name'=>'a_shape_options','min'=>'','max'=>'','step'=>'','inputClass' =>  'is-required'],
                            'a_increased_by'  =>  ['label' =>  '','type'=>'dropdown','values'=>[''=>''],'symbol'=>'mm','step'=>'0.5','min'=>0,'max'=>'99','round'=>'1','pad'=>'','class'=>'col-md-6 col-xs-6','data-name'=>'a_increased_by','increment'=>0.5,'inputClass' =>  'is-required']
                        ]
                    ],
                    'ms_b' =>  [
                        'label' =>  'B',
                        'values'    =>  [
                            'b_shape_options'  =>  ['label' =>  '','type'=>'dropdown','values'=>['Increase By:'=>'Increase By:','Decrease By:'=>'Decrease By:','Make:'=>'Make:'],'step'=>'0.5','class'=>'col-md-6 col-xs-6','data-name'=>'b_shape_options','min'=>'','max'=>'','step'=>'','inputClass' =>  'is-required'],
                            'b_increased_by'  =>  ['label' =>  '','type'=>'dropdown','symbol'=>'mm','step'=>'0.5','min'=>0,'max'=>'99','round'=>'1','pad'=>'','class'=>'col-md-6 col-xs-6','values'=>[''=>''],'data-name'=>'b_increased_by','increment'=>0.5,'inputClass' =>  'is-required']
                        ]
                    ],
                ],
                'specify_thickness'  =>  [
                    'label'=>'Specify Thickness',
                    'ddl_values'    =>  ['Normal','Industrial','Specify'],
                    'values'=>[
                        ['label'=>'Safety Thickness','show'=>false,'name'=>'safety_thickness'],
                        ['label'=>'Safety Stamp','show'=>false,'name'=>'safety_stamp']
                    ]
                ],
                'edge_polish'  =>  [
                    'label'=>'Polish Edges',
                ],
                'bevel_type'  =>  [
                    'normal' =>  [
                        'label' =>   'Normal'
                    ],
                    'flush_front' =>  [
                        'label' =>   'Flush Front'
                    ],
                    'split_bevel' =>  [
                        'label' =>   'Split Bevel'
                    ],
                ],
                'invoice_order_no'  =>  [
                    'label' =>  'Invoice/Order Number'
                ],
                'ft_other'  =>  [
                    'label' =>  'Other'
                ]
            ],
            'lens_information'  =>  [
                'aspheric'  =>  [
                    'label'=>'Aspheric',
                ],
                'lens_manufacturer'  =>  [
                    // 'essilor_natural'   =>  'Essilor Natural',
                    // 'essilor'   =>  'Essilor'
                ],
                'lens_name'  =>  [
                    // 'flat_top_28'   =>  'Flat Top 28',
                    // 'distinctive'   =>  'Distinctive'
                ],
                'sub_type'  =>  [
                    // 'flat_top_28'   =>  'Flat Top 28',
                    // 'distinctive'   =>  'Distinctive'
                ],
                'base_curve'  =>  [
                    // '2.25'   =>  '2.25',
                    // '4.25'   =>  '4.25',
                    // '6.25'  =>  '6.25'
                ],
                'lens_material'  =>  [
                    // 'flat_top_28'   =>  'Plastic 1.5',
                    // 'distinctive'   =>  'Polycarbonate 1.59'
                ],
                'lens_filter'  =>  [
                    // 'clear'   =>  'Polarized',
                    // 'distinctive'   =>  'Tinted'
                ],
                'lens_colour'  =>  [
                    // 'clear'   =>  'Grey',
                    // 'distinctive'   =>  'Brown'
                ],
                'coating_made'  =>  [
                ],
                'coating_name'  =>  [
                ],
                'coating_colour'  =>  [
                ],
            ],
            'additional_items'  =>  [
                'add_clip_on'  =>  [
                    'label'=>'Add Clip-on',
                    'values'    =>  [
                        'Nasal Attachment with Tinted Lenses'       =>  'Nasal Attachment with Tinted Lenses',
                        'Nasal Attachment with Polarized Lenses'    =>  'Nasal Attachment with Polarized Lenses',
                        'Classic with Bar with Tinted Lenses'       =>  'Classic with Bar with Tinted Lenses',
                        'Classic with Bar with Polarized Lenses'    =>  'Classic with Bar with Polarized Lenses'
                    ]
                ],
                'add_repairs'  =>  [
                    'label'=>'Add Repairs'
                ],
            ],
            'special_instructions'  =>  ['label'    =>  'Special Instructions']
        ];
    }

    public static function getCombinationHideArray(){
        return [
            'lens_type' => [
                'Single_Vision_Finished'    => [
                    'hide'  =>  ['right-strength','left-strength','right-boc','left-boc','right-corridor','left-corridor','panto','wrap','vertex','specify_thickness-content'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_oc','left_oc','right_npd','left_npd'],
                    'required'  =>  []
                ],
                'Single_Vision_Surfaced_Conventional'    => [
                    'hide'  =>  ['right-strength','left-strength','right-boc','left-boc','right-corridor','left-corridor','panto','wrap','vertex'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_oc','left_oc','right_npd','left_npd'],
                    'required'  =>  []
                ],
                'Single_Vision_Surfaced_Digital_HD'    => [
                    'hide'  =>  ['right-strength','left-strength','right-boc','left-boc','right-corridor','left-corridor','panto','wrap','vertex'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_oc','left_oc','right_npd','left_npd'],
                    'required'  =>  []
                ],
                'Single_Vision_AntiminusFatigue'    => [
                    'hide'  =>  ['right-add','left-add','left-npd','right-npd','right-intermediate_strength','left-intermediate_strength','right-intermediate_pd','left-intermediate_pd','left-boc','right-boc','left-corridor','right-corridor','panto','wrap','vertex'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_npd','left_npd'],
                    'required'  =>  []
                ],
                'Lined_Multiminusfocal'    => [
                    'hide'  =>  ['right-strength','left-strength','right-intermediate_strength','left-intermediate_strength','right-intermediate_pd','left-intermediate_pd','left-corridor','right-corridor','panto','wrap','vertex'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_npd','left_npd',],
                    'required'  =>  []
                ],
                'Progressive'    => [
                    'hide'  =>  ['right-strength','left-strength','left-npd','right-npd','right-intermediate_strength','left-intermediate_strength','left-boc','right-boc','right-intermediate_pd','left-intermediate_pd'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_npd','left_npd'],
                    'required'  =>  []
                ],
                'Computer_Progressive'    => [
                    'hide'  =>  ['left-strength','left-npd','right-npd','right-intermediate_pd','left-intermediate_pd','right-boc','left-boc','right-corridor','left-corridor','panto','wrap','vertex','right-intermediate_strength','left-intermediate_strength'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_npd','left_npd'],
                    'required'  =>  []
                ],
                'Extended_Computer_Progressive'    => [
                    'hide'  =>  ['left-strength','left-npd','right-npd','right-intermediate_pd','left-intermediate_pd','right-boc','left-boc','right-corridor','left-corridor','panto','wrap','vertex','right-intermediate_strength','left-intermediate_strength'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_npd','left_npd'],
                    'required'  =>  []
                ],
                'Special'    => [
                    'hide'  =>  ['right-strength','left-strength','right-intermediate_strength','left-intermediate_strength','left-intermediate_pd','right-intermediate_pd','right-corridor','left-corridor','panto','wrap','vertex'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_npd','left_npd'],
                    'required'  =>  ['right_oc','left_oc']
                ],
                'Myopia_Management'    => [
                    'hide'  =>  ['right-boc','left-boc','right-add','left-add','right-npd','left-npd','right-strength','left-strength','right-intermediate_strength','left-intermediate_strength','left-intermediate_pd','right-intermediate_pd','right-corridor','left-corridor','panto','wrap','vertex'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ]
                // 'sv'    =>  [
                //     'hide'  =>  ['specify_boc_height-container','remove_specify_boc_height','right-boc','left-boc','right-sh','left-sh','special_info','use_shorter_corridor','aspheric','final_rx'],
                //     'show'  =>   ['sv','left-oc','right-oc','left-intermediate_pd','right-intermediate_pd','right-npd','left-npd','add_for_reading','left-pd','right-pd','right-intermediate_strength','left-intermediate_strength'],
                //     'options'   =>  [
                //         'sv_distance'    =>  [
                //             'hide'  =>  ['add_for_reading','right-intermediate_pd','left-intermediate_pd','left-npd','right-npd'],
                //             'show'  =>   []
                //         ],
                //         'sv_intermediate_or_computer'    =>  [
                //             'hide'  =>  ['left-pd','right-pd','left-npd','right-npd'],
                //             'show'  =>   ['final_rx']
                //         ],
                //         'sv_reading'    =>  [
                //             'hide'  =>  ['left-pd','right-pd','left-intermediate_pd','right-intermediate_pd','right-intermediate_strength','left-intermediate_strength'],
                //             'show'  =>   ['final_rx']
                //         ],
                //     ]
                // ],
                // 'multi_focal'    =>  [
                //     'hide'  =>  ['left-oc','right-oc','left-intermediate_pd','right-intermediate_pd','special_info','use_shorter_corridor','final_rx','right-intermediate_strength','left-intermediate_strength'],
                //     'show'  =>   ['specify_boc_height-container','remove_specify_boc_height','right-boc','left-boc','right-sh','left-sh','aspheric','add_for_reading','left-npd','right-npd','left-pd','right-pd']
                // ],
                // 'progressive'    =>  [
                //     'hide'  =>  ['left-oc','right-oc','specify_boc_height-container','remove_specify_boc_height','right-boc','left-boc','left-intermediate_pd','right-intermediate_pd','final_rx'],
                //     'show'  =>   ['progressive','right-sh','left-sh','aspheric','special_info','use_shorter_corridor','right-npd','left-npd','add_for_reading','left-pd','right-pd','right-intermediate_strength','left-intermediate_strength'],
                //     'options'   =>  [
                //         'progressive_hd_digital'    =>  [
                //             'hide'  =>  ['special_info','right-npd','left-npd'],
                //             'show'  =>   []
                //         ],
                //         'progressive_full_custom_fit'    =>  [
                //             'hide'  =>  [],
                //             'show'  =>   ['special_info']
                //         ],
                //         'progressive_intermediate_computer_reading'    =>  [
                //             'hide'  =>  ['use_shorter_corridor','special_info'],
                //             'show'  =>   []
                //         ],
                //         'progressive_conventional'    =>  [
                //             'hide'  =>  ['special_info','use_shorter_corridor','right-npd','left-npd'],
                //             'show'  =>   []
                //         ]
                //     ]
                // ]
            ],
            'sub_type'  =>  [
                'Distance'    => [
                    'hide'  =>  ['left-add','right-add','left-npd','right-npd','left-intermediate_pd','right-intermediate_pd','right-intermediate_strength','left-intermediate_strength'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'Intermediate_(Lab_Converts_Rx)'    => [
                    'hide'  =>  ['right-pd','left-pd','left-npd','right-npd'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'Intermediate_(PreminusConverted_Rx)'    => [
                    'hide'  =>  ['left-add','right-add','left-npd','right-npd','right-pd','left-pd','right-intermediate_strength','left-intermediate_strength'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'Reading_(Lab_Converts_Rx)'    => [
                    'hide'  =>  ['left-intermediate_pd','right-intermediate_pd','right-pd','left-pd','right-intermediate_strength','left-intermediate_strength'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  ['right_npd','left_npd']
                ],
                'Reading_(PreminusConverted_Rx)'    => [
                    'hide'  =>  ['left-add','right-add','left-intermediate_pd','right-intermediate_pd','right-pd','left-pd','right-intermediate_strength','left-intermediate_strength'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  ['right_npd','left_npd']
                ],
                'Digital_HD_Freeform'    => [
                    'hide'  =>  ['left-npd','right-npd'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'Surfaced_Conventional'    => [
                    'hide'  =>  ['right-boc','left-boc','left-npd','right-npd','right-corridor','left-corridor','panto','wrap','vertex'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'Omega_SV'    => [
                    'hide'  =>  ['right-npd','left-npd','left-add','right-add','right-npd','left-npd','right-boc','left-boc'],
                    'show'  =>   [],
                    'not_required'  =>  ['right_oc','left_oc'],
                    'required'  =>  []
                ],
                'Myopia_Management'    => [
                    'hide'  =>  ['right-npd','left-npd','right-boc','left-boc','right-add','left-add'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ]
            ],
            'job_type' =>  [
                'uncut'  => [
                    'hide'  =>  ['pof-container'],
                    'show'  =>   ['frame_info'],
                    'not_required'  =>  ['brand','model','colour','size'],
                    'required'  =>  []
                ],
                'edged_frame_to_follow'  => [
                    'hide'  =>  [],
                    'show'  =>   ['edge_polish','frame_info'],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'edged_no_frame_to_follow'  => [
                    'hide'  =>  [],
                    'show'  =>   ['invoice_order_no','edge_polish','frame_info'],
                    'not_required'  =>  ['brand','model','colour','size'],
                    'required'  =>  []
                ],
                'lab_supplies_frame'  => [
                    'hide'  =>  ['pof-container'],
                    'show'  =>   ['frame_info','edge_polish'],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
            ],
            'frame_type'    =>  [
                'full_metal' =>  [
                    'hide'  =>  ['ft_other'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'full_plastic' =>  [
                    'hide'  =>  ['ft_other'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'nyloon_groove' =>  [
                    'hide'  =>  ['ft_other'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'rimless_drill' =>  [
                    'hide'  =>  ['ft_other'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'metal_groove' =>  [
                    'hide'  =>  ['ft_other'],
                    'show'  =>   [],
                    'not_required'  =>  [],
                    'required'  =>  []
                ],
                'other' =>  [
                    'hide'  =>  [],
                    'show'  =>   ['ft_other'],
                    'not_required'  =>  [],
                    'required'  =>  []
                ]
            ]
        ];
    }
    public static function getBaseArray(){
        return [
            'up'    =>  'Up',
            'down'  =>  'Down',
            'in'    =>  'In',
            'out'   =>  'Out'
        ];
    }

    public function getJobType($type) {

        switch ($type) {
            case 'uncut':
                return 'Uncut';
                break;

            case 'edged_frame_to_follow':
                return 'Edged (Frame to Follow)';
                break;

            case 'edged_no_frame_to_follow':
                return 'Edged (From Shape, No frame to follow)';
                break;
        }
    }

    public function getFrameType($type) {

        switch ($type) {
            case 'full_metal':
                return 'Full Metal';
                break;

            case 'full_plastic':
                return 'Full Plastic';
                break;

            case 'nyloon_groove':
                return 'Nylon Groove';
                break;

            case 'rimless_drill':
                return 'Rimless (Drill / Notch)';
                break;

            case 'metal_groove':
                return 'Metal Groove';
                break;

            case 'other':
                return 'Other';
                break;
            default:
                return RxValidations::decodeValue($type);
                break;
        }
    }

    public function roundNumber($number, $decimal) {
        if (is_numeric($number)) {
            return number_format((float)$number, $decimal, '.', '');
        }
        return '';
    }


    public function addNumberSign($number){
        if($number != ''){
            if ($number > 0) {
                if (strpos($number, '+') === false) {
                    $number = "+$number"; // Output: +10
                }
            } elseif ($number < 0) {
                if (strpos($number, '-') === false) {
                    $number = "$number"; // Output: -10
                }
            } else {
                $number = '+0.00'; // Output: 0
            }
        }
        return $number;
    }

    public function getShowData(){
        $data = json_decode($this->data,true);

        $lensType = isset($data['lens']['lens_type']) ? $data['lens']['lens_type'] : '';
        $lensFilter = isset($data['lens']['lens_filter']) ? $data['lens']['lens_filter'] : '';
        if(
            isset($data['rx']['extended_info']['right_base_curve']) && 
            !empty($data['rx']['extended_info']['right_base_curve']) && 
            $lensType == 'Single_Vision_Finished' && 
            stripos($lensFilter, 'plano') === false
        ){
            $data['rx']['extended_info']['right_base_curve'] = '';
        }

        $jobType =  isset($data['frame']['job_type']) ? RxValidations::decodeValue($data['frame']['job_type']) : '';
        $jobTypeRaw =  isset($data['frame']['job_type']) ? $data['frame']['job_type'] : '';
        if(isset($data['frame']['job_type']) && ($data['frame']['job_type'] == 'edged_frame_to_follow' || $data['frame']['job_type'] == 'lab_supplies_frame') && isset($data['frame']['frame_info']['brand']) && !empty($data['frame']['frame_info']['brand'])){
            // $jobType .= ' ( '.$data['frame']['frame_info']['brand'].' '.$data['frame']['frame_info']['model'].' '.$data['frame']['frame_info']['colour'].' ) ';
            // if(isset($data['frame']['frame_info']['pof']) && $data['frame']['frame_info']['pof'] && $data['frame']['job_type'] == 'edged_frame_to_follow'){
            //     $jobType .= ' ✓ POF';
            // }
        }elseif(isset($data['frame']['job_type']) && $data['frame']['job_type'] == 'edged_no_frame_to_follow'){
            $jobType = 'Edged from Shape #'.$data['frame']['invoice_order_no'];
        }
        $rightOC = '';
        $leftOC = '';
        if(isset($data['rx']['distance']['right_oc'])){
            $rightOC = self::roundNumber($data['rx']['distance']['right_oc'],2);
        }elseif(isset($data['rx']['extended_info']['right_oc'])){
            $rightOC = self::roundNumber($data['rx']['extended_info']['right_oc'],2);
        }
        if(isset($data['rx']['distance']['left_oc'])){
            $leftOC = self::roundNumber($data['rx']['distance']['left_oc'],2);
        }elseif(isset($data['rx']['extended_info']['left_oc'])){
            $leftOC = self::roundNumber($data['rx']['extended_info']['left_oc'],2);
        }
        $rpd_ipd['label'] = 'PD';
        $rpd_ipd['value'] = isset($data['rx']['extended_info']['right_pd']) ? self::roundNumber($data['rx']['extended_info']['right_pd'],1) : '';
        if(empty($rpd_ipd['value']) && isset($data['rx']['extended_info']['right_intermediate_pd']) && !empty($data['rx']['extended_info']['right_intermediate_pd'])){
            $rpd_ipd['label'] = 'IPD';
            $rpd_ipd['value'] = isset($data['rx']['extended_info']['right_intermediate_pd']) ? self::roundNumber($data['rx']['extended_info']['right_intermediate_pd'],1) : '';
        }
        $lpd_ipd['label'] = 'PD';
        $lpd_ipd['value'] = isset($data['rx']['extended_info']['left_pd']) ? self::roundNumber($data['rx']['extended_info']['left_pd'],1) : '';
        if(empty($lpd_ipd['value']) && isset($data['rx']['extended_info']['left_intermediate_pd']) && !empty($data['rx']['extended_info']['left_intermediate_pd'])){
            $lpd_ipd['label'] = 'IPD';
            $lpd_ipd['value'] = isset($data['rx']['extended_info']['left_intermediate_pd']) ? self::roundNumber($data['rx']['extended_info']['left_intermediate_pd'],1) : '';
        }
        $intermediateStrengthText = '';
        $rightSph = isset($data['rx']['distance']['right_sph']) ? self::addNumberSign(self::roundNumber( $data['rx']['distance']['right_sph'],2)) : '';
        $leftSph = isset($data['rx']['distance']['left_sph']) ? self::addNumberSign(self::roundNumber( $data['rx']['distance']['left_sph'],2)) : '';
        $rightAdd = isset($data['rx']['add_for_reading']['right_add']) ? self::addNumberSign(self::roundNumber($data['rx']['add_for_reading']['right_add'],2)) : '';
        $leftAdd = isset($data['rx']['add_for_reading']['left_add']) ? self::addNumberSign(self::roundNumber($data['rx']['add_for_reading']['left_add'],2)) : '';
        if(isset($data['lens']['sub_type']) && !empty($data['lens']['sub_type'])){
            if($data['lens']['sub_type'] == 'Intermediate_(Lab_Converts_Rx)'){
                if(!empty($rightSph) && isset($data['rx']['add_for_reading']['right_intermediate_strength'])){
                    $newAdd = $rightSph + ($rightAdd * ($data['rx']['add_for_reading']['right_intermediate_strength']/100));
                    $rightSph = self::addNumberSign(roundToNearestQuarter(self::roundNumber($newAdd,2)));
                    $intermediateStrengthText = 'Rx converted based on Intermediate Strength '.$data['rx']['add_for_reading']['right_intermediate_strength'].'%';
                }
                if(!empty($leftSph) && isset($data['rx']['add_for_reading']['right_intermediate_strength'])){
                    $newAdd = $leftSph + ($leftAdd * ($data['rx']['add_for_reading']['right_intermediate_strength']/100));
                    $leftSph = self::addNumberSign(roundToNearestQuarter(self::roundNumber($newAdd,2)));
                    $intermediateStrengthText = 'Rx converted based on Intermediate Strength '.$data['rx']['add_for_reading']['right_intermediate_strength'].'%';
                }
            }elseif($data['lens']['sub_type'] == 'Reading_(Lab_Converts_Rx)'){
                if(!empty($rightAdd) && !empty($rightSph)){
                    $intermediateStrengthText = 'Rx converted based on Add Power '.$rightAdd;
                    $rightSph = self::addNumberSign(self::roundNumber($rightSph + $rightAdd,2));
                }
                if(!empty($leftAdd) && !empty($leftSph)){
                    $intermediateStrengthText = 'Rx converted based on Add Power '.$leftAdd;
                    $leftSph = self::addNumberSign(self::roundNumber($leftSph + $leftAdd,2));
                }
            }
        }
        $rightCTLabel = (!is_null($rightSph) && $rightSph > 0) ? 'R: ET' : 'R: CT';
        $leftCTLabel  = (!is_null($leftSph) && $leftSph > 0) ? 'L: ET' : 'L: CT';

        $bevelType = isset($data['frame']['bevel_type']) ? RxValidations::decodeValue($data['frame']['bevel_type']) : '';
        $frameTypeRaw = isset($data['frame']['frame_type']) ? $data['frame']['frame_type'] : '';
        if($jobTypeRaw != 'uncut' && ($frameTypeRaw == 'full_metal' || $frameTypeRaw == 'full_plastic')){

        }else{
            $bevelType = '';
        }

        $response = [
            'name'    =>  $data['first_name'].' '.$data['last_name'],
            'intermediateStrengthText' =>  $intermediateStrengthText,
            'package'   =>  [
                'label'  =>  isset($data['package_label']) ? $data['package_label'] : '',
                'price'  =>  isset($data['package_price']) ? $data['package_price'] : '',
                'frame_price'  =>  isset($data['frame_price']) ? $data['frame_price'] : '',
            ],
            'distance'  =>  [
                'label' =>  [
                    'Sph','Cyl','Axis','Prism',''
                ],
                'right_dist_type'   =>  isset($data['rx']['distance']['right_dist_type']) ? $data['rx']['distance']['right_dist_type'] : '',
                'right' =>  [
                    // 'type'   =>  isset($data['rx']['distance']['right_dist_type']) ? $data['rx']['distance']['right_dist_type'] : '',
                    'sph'   =>  $rightSph,
                    'cyl'   =>  isset($data['rx']['distance']['right_cyl']) ? self::addNumberSign(self::roundNumber( $data['rx']['distance']['right_cyl'],2)) : '',
                    'axis'   =>  isset($data['rx']['distance']['right_axis']) && !empty($data['rx']['distance']['right_axis']) ? str_pad($data['rx']['distance']['right_axis'], 3, '0', STR_PAD_LEFT) : '',
                    // 'add'   =>  isset($data['rx']['add_for_reading']['right_add']) ? self::addNumberSign(self::roundNumber($data['rx']['add_for_reading']['right_add'],2)) : '',
                    'prism1'   =>  isset($data['rx']['prism']['right_prism1']) ? self::roundNumber($data['rx']['prism']['right_prism1'],2).' '.ucfirst($data['rx']['prism']['right_base1']).' '.self::roundNumber($data['rx']['prism']['right_prism2'],2).' '.ucfirst($data['rx']['prism']['right_base2']): '',
                    ''  =>  ''
                ],
                'left_dist_type'   =>  isset($data['rx']['distance']['left_dist_type']) ? $data['rx']['distance']['left_dist_type']  : '',
                'left' =>  [
                    'sph'   =>  $leftSph,
                    'cyl'   =>  isset($data['rx']['distance']['left_cyl']) ? self::addNumberSign(self::roundNumber($data['rx']['distance']['left_cyl'],2)) : '',
                    'axis'   =>  isset($data['rx']['distance']['left_axis']) && !empty($data['rx']['distance']['left_axis'])  ? str_pad($data['rx']['distance']['left_axis'], 3, '0', STR_PAD_LEFT) : '',
                    // 'add'   =>  isset($data['rx']['add_for_reading']['left_add']) ? self::addNumberSign(self::roundNumber($data['rx']['add_for_reading']['left_add'],2)) : '',
                    'prism1'   =>  isset($data['rx']['prism']['left_prism1']) && isset($data['rx']['prism']['left_base1']) ? self::roundNumber($data['rx']['prism']['left_prism1'],2).' '.ucfirst($data['rx']['prism']['left_base1']).' '.self::roundNumber($data['rx']['prism']['left_prism2'],2).' '.ucfirst($data['rx']['prism']['left_base2']): '',
                    ''  =>  ''
                ],
            ],
            'single_info' => [
                'label' =>  ['Corridor','Boost',
                            // 'IS',
                            'Panto','Wrap','Vertex',
                            // 'BC'
                        ],
                'values' =>[
                    'corridor'   =>  isset($data['rx']['extended_info']['right_corridor']) ? $data['rx']['extended_info']['right_corridor'] : '',
                    'strength'   =>  isset($data['rx']['distance']['right_strength']) ? $data['rx']['distance']['right_strength'] : '',
                    // 'intermediate_strength'   =>  isset($data['rx']['distance']['right_intermediate_strength']) ?? $data['rx']['distance']['right_intermediate_strength'],
                    'pento' =>  isset($data['rx']['special_info']['panto']) ? $data['rx']['special_info']['panto'] : '',
                    'wrap'  =>  isset($data['rx']['special_info']['wrap']) ? $data['rx']['special_info']['wrap'] : '',
                    'vertex'  =>  isset($data['rx']['special_info']['vertex']) ? $data['rx']['special_info']['vertex'] : '',
                    // 'base_curve'   =>  isset($data['rx']['extended_info']['right_base_curve']) ? $data['rx']['extended_info']['right_base_curve'] : '',
                ]
            ],
            'reading'   =>  [
                'label' =>  [
                    'Add','Height',$rpd_ipd['label'],
                    // 'IPD',
                    'NPD','BOC'
                    // ,'',''
                ],
                'right' =>[
                    'add'   =>  empty($intermediateStrengthText) && isset($data['rx']['add_for_reading']['right_add']) ? self::addNumberSign(self::roundNumber($data['rx']['add_for_reading']['right_add'],2)) : '',
                    'fh'   =>   $rightOC,
                    'pd'   =>   $rpd_ipd['value'],
                    // 'intermediate_pd'   =>   isset($data['rx']['extended_info']['right_intermediate_pd']) ? self::roundNumber($data['rx']['extended_info']['right_intermediate_pd'],1) : '',
                    'npd'   =>   isset($data['rx']['extended_info']['right_npd']) ? self::roundNumber($data['rx']['extended_info']['right_npd'],1) : '',
                    'boc'   =>  isset($data['rx']['add_for_reading']['right_boc']) ? $data['rx']['add_for_reading']['right_boc'] : '',
                    // '1'  =>  '',
                    // '2'  =>  ''
                ],
                'left' =>[
                    'add'   =>  empty($intermediateStrengthText) && isset($data['rx']['add_for_reading']['left_add']) ? self::addNumberSign(self::roundNumber($data['rx']['add_for_reading']['left_add'],2)) : '',
                    'fh'   =>   $leftOC,
                    'pd'   =>   $lpd_ipd['value'],
                    // 'intermediate_pd'   =>   isset($data['rx']['extended_info']['left_intermediate_pd']) ? self::roundNumber($data['rx']['extended_info']['left_intermediate_pd'],1) : '',
                    'npd'   =>   isset($data['rx']['extended_info']['left_npd']) ? self::roundNumber($data['rx']['extended_info']['left_npd'],1) : '',
                    'boc'   =>  isset($data['rx']['add_for_reading']['left_boc']) ? $data['rx']['add_for_reading']['left_boc'] : '',
                    // '1'  =>  '',
                    // '2'  =>  ''
                ]
            ],
            'frame'     =>  [
                'job_type'      =>  $jobType,
                'job_type_key'      =>  isset($data['frame']['job_type']) ? $data['frame']['job_type'] : '',
                'frame_type'    =>  isset($data['frame']['frame_type']) ? $this->getFrameType($data['frame']['frame_type']) : '',
                'frame_type_key'    =>  isset($data['frame']['frame_type']) ? $data['frame']['frame_type'] : '',
                'o_frame_type'    =>  isset($data['frame']['frame_type']) ? str_replace(' ','_',$data['frame']['frame_type']) : '',
                'bevel_type'    =>  $bevelType,
                'specify_thickness'    =>  isset($data['frame']['specify_thickness']) && isset($data['frame']['specify_thickness']['value']) ? RxValidations::decodeValue($data['frame']['specify_thickness']['value']) : '',
                'frame_size'    =>  [
                    'label' =>  ['A','B','ED','DBL'],
                    'values' =>[
                        0 =>  isset($data['frame']['frame_size']['a']) ? self::roundNumber($data['frame']['frame_size']['a'],1) : '',
                        1  =>  isset($data['frame']['frame_size']['b']) ? self::roundNumber($data['frame']['frame_size']['b'],1) : '',
                        2  =>  isset($data['frame']['frame_size']['ed']) ? self::roundNumber($data['frame']['frame_size']['ed'],1) : '',
                        3  =>  isset($data['frame']['frame_size']['dbl']) ? self::roundNumber($data['frame']['frame_size']['dbl'],1) : ''
                    ],
                ],
                'modify_shape'  =>  [
                    'a' =>  (isset($data['frame']['modify_shape']['ms_a']) && $data['frame']['modify_shape']['ms_a']) ? '✓' : '',
                    'a_option'  =>  isset($data['frame']['modify_shape']['a_shape_options']) ? $data['frame']['modify_shape']['a_shape_options'] : '',
                    'a_increase'  =>  isset($data['frame']['modify_shape']['a_increased_by']) ? $data['frame']['modify_shape']['a_increased_by'] : '',
                    'b' =>  (isset($data['frame']['modify_shape']['ms_b']) && $data['frame']['modify_shape']['ms_b']) ? '✓' : '',
                    'b_option'  =>  isset($data['frame']['modify_shape']['b_shape_options']) ? $data['frame']['modify_shape']['b_shape_options'] : '',
                    'b_increase'  =>  isset($data['frame']['modify_shape']['b_increased_by']) ? $data['frame']['modify_shape']['b_increased_by'] : ''
                ],
                'specify_thickness_data' => [
                    'r_ct_or_et' => isset($data['frame']['specify_thickness']['r_ct_or_et']) 
                        ? $rightCTLabel . ' ' . number_format($data['frame']['specify_thickness']['r_ct_or_et'], 1)
                        : '',
                    'l_ct_or_et' => isset($data['frame']['specify_thickness']['l_ct_or_et']) 
                        ? $leftCTLabel . ' ' . number_format($data['frame']['specify_thickness']['l_ct_or_et'], 1)
                        : '',
                ],
                // 'specify_thickness' => (isset($data['frame']['specify_thickness']) && $data['frame']['specify_thickness']) ? '✓' : '',
                'safety_thickness' => (isset($data['frame']['safety_thickness']) && $data['frame']['safety_thickness']) ? '✓' : '',
                'safety_stamp' => (isset($data['frame']['safety_stamp']) && $data['frame']['safety_stamp']) ? '✓ Safety Stamp' : '',
                'edge_polish' => (isset($data['frame']['edge_polish']) && $data['frame']['edge_polish']) ? ', ✓ Polish Edges' : '',
            ],
            'lens'  =>  [
                'lens_type'    =>  isset($data['lens']['lens_type']) ? RxValidations::decodeValue($data['lens']['lens_type']) : '',
                'sub_type'    =>  isset($data['lens']['sub_type']) ? RxValidations::decodeValue($data['lens']['sub_type']) : '',
                'lens_manufacturer'    =>  isset($data['lens']['lens_manufacturer']) ? RxValidations::decodeValue($data['lens']['lens_manufacturer']) : '',
                'lens_name'    =>  isset($data['lens']['lens_name']) ? RxValidations::decodeValue($data['lens']['lens_name']) : '',
                'base_curve'    =>  isset($data['lens']['base_curve']) ? RxValidations::decodeValue($data['lens']['base_curve']) : '',
                'lens_material'    =>  isset($data['lens']['lens_material']) ? RxValidations::decodeValue($data['lens']['lens_material']) : '',
                'lens_filter'    =>  isset($data['lens']['lens_filter']) ? RxValidations::decodeValue($data['lens']['lens_filter']) : '',
                'lens_colour'    =>  isset($data['lens']['lens_colour']) ? RxValidations::decodeValue($data['lens']['lens_colour']) : '',
                'coating_made'    =>  isset($data['lens']['coating_made']) ? RxValidations::decodeValue($data['lens']['coating_made']) : '',
                'coating_name'    =>  isset($data['lens']['coating_name']) ? RxValidations::decodeValue($data['lens']['coating_name']) : '',
                'coating_colour'    =>  isset($data['lens']['coating_colour']) ? RxValidations::decodeValue($data['lens']['coating_colour']) : '',
            ],
            'additional_items'  =>  [
                'add_clip_on'   =>  isset($data['additional_items']['clip_on_type']) ? 'Custom Clip-on / '.RxValidations::decodeValue($data['additional_items']['clip_on_type']).' / '.RxValidations::decodeValue($data['additional_items']['clip_on_lens_type']).' / '.RxValidations::decodeValue($data['additional_items']['clip_on_lens_colour']) : '',
                'repairs_services'   =>  isset($data['additional_items']['repairs_services']) ? $data['additional_items']['repairs_services'] : [],
                'repair_description'   =>  isset($data['additional_items']['repair_description']) ? $data['additional_items']['repair_description'] : '',
                'special_instructions'   =>  $data['special_instructions']
            ]
        ];
        if(isset($data['rx']['extended_info']['right_base_curve']) && !empty($data['rx']['extended_info']['right_base_curve'])){
            $response['single_info']['label'][] = 'BC';
            $response['single_info']['values']['base_curve']= $data['rx']['extended_info']['right_base_curve'];
        }
        if(isset($data['frame']['frame_size']['ed_or_shape']) && $data['frame']['frame_size']['ed_or_shape'] == 'shape'){
            $response['frame']['frame_size']['label'][2] = 'Shape';
            $response['frame']['frame_size']['values']['ed'] = strtoupper($data['frame']['frame_size']['shape']);
        }

        if(isset($data['rx']['special_info']) && !empty($data['rx']['special_info']['pento'])){
            $response['s_info'] = [
                'label' =>  ['Panto','Wrap','Vertex'],
                'values' =>[
                    'pento' =>  isset($data['rx']['special_info']['panto']) ? $data['rx']['special_info']['panto'] : '',
                    'wrap'  =>  isset($data['rx']['special_info']['wrap']) ? $data['rx']['special_info']['wrap'] : '',
                    'vertex'  =>  isset($data['rx']['special_info']['vertex']) ? $data['rx']['special_info']['vertex'] : ''
                ],
                'use_shorter_corridor'  =>  (isset($data['rx']['use_shorter_corridor'])) ? $data['rx']['use_shorter_corridor'] : false
            ];
        }
        // if($data['frame']['job_type'] == 'edged_frame_to_follow'){
            $response['frame']['frame_info'] = [
                'label' =>  ['Brand','Model','Colour','POF','safety stamp','polish edges', 'out edge'],
                'values' =>[
                    'brand' =>  isset($data['frame']['frame_info']['brand']) ? $data['frame']['frame_info']['brand'] : '',
                    'model'  =>  isset($data['frame']['frame_info']['model']) ? $data['frame']['frame_info']['model'] : '',
                    'colour'  =>  isset($data['frame']['frame_info']['colour']) ? $data['frame']['frame_info']['colour'] : '',
                    'size'  =>  isset($data['frame']['frame_info']['size']) ? $data['frame']['frame_info']['size'] : '',
                    'pof'   =>  (isset($data['frame']['frame_info']['pof']) && $data['frame']['frame_info']['pof']) ? '✓' : '',
                    'safety stamp' => (isset($data['frame']['safety_stamp']) && $data['frame']['safety_stamp']) ? '✓' : '',
                    'polish edges' => (isset($data['frame']['edge_polish']) && $data['frame']['edge_polish']) ? '✓' : '',
                    'out edge' => (isset($data['frame']['out_edge']) && $data['frame']['out_edge']) ? '<img src="' . generateBarcode('out_edge', '25', '3') . '" width="70px">' : '',
                    // 'frame upc' => (isset($data['frame']['job_type']) && $data['frame']['job_type'] == 'lab_supplies_frame') ? '<img src="' . generateBarcode($data['frame']['upc'], '25', '3') . '" width="70px">' : '',
                ],
            ];
        // }
        if(isset($data['frame']['job_type']) && $data['frame']['job_type'] == 'edged_no_frame_to_follow')
            $response['frame']['invoice_order_no'] = $data['frame']['invoice_order_no'];

        if(isset($data['group_values']) && !empty($data['group_values'])){
            if(isset($data['lens']['lens_type'])){
                $data['att_values'] = $data['lens']['lens_type'].'####'.$data['att_values'];
            }
            $response['lens_group'] = [
                'group_values'  =>  explode("####", trim($data['group_values'])),
                'att_values'  =>  explode("####", trim($data['att_values'])),
            ];
        }
        $response['multipair_ref']   =   '';
        if(isset($data['multipair_ref']) && !empty($data['multipair_ref'])){
            $response['multipair_ref'] = $data['multipair_ref'];
        }
        if(isset($data['is_tinting'])){
            $response['is_tinting'] = $data['is_tinting'];
            $response['tint_option'] = $data['tint_option'];
            $response['tint_color_sample'] = $data['tint_color_sample'];
            $response['tinting'] = $data['tinting'];
            $response['edging'] = $data['edging'];
        }
        $response['prices'] = [
            'right_lens_type'   =>  '',
            'left_lens_type'   =>  '',
            'custom_clip_on'   =>  '',
            'additional'   =>  [],
            'repairs_services'   =>  [],
            'total_price'   =>  ''
        ];
        if(isset($data['prices'])){
            $response['prices'] = [
                'right_lens_type'   =>  isset($data['prices']['right_lens_type']) ? $data['prices']['right_lens_type'] : '',
                'left_lens_type'   =>  isset($data['prices']['left_lens_type']) ? $data['prices']['left_lens_type'] : '',
                'custom_clip_on'   =>  isset($data['prices']['custom_clip_on']) ? $data['prices']['custom_clip_on'] : '',
                'additional'   =>  isset($data['prices']['additional']) ? $data['prices']['additional'] : [],
                'repairs_services'   =>  isset($data['prices']['repairs_services']) ? $data['prices']['repairs_services'] : [],
                'total_price'   =>  isset($data['prices']['total_price']) ? $data['prices']['total_price'] : ''
            ];
        }

        // echo "<pre>";
        // print_r($data);
        // print_r($response);
        // exit;

        return $response;
    }

    public static function getDefaultArray($data){
        return [
            'price' =>  0,
            'first_name'    =>  $data->patient,
            'last_name'     =>  '',
            'special_instructions'  =>  $data->special_instruction,
            'lens'  =>   [
                'lens_type' =>  $data->lens_title,
                'lens_type2'    =>  '',
                'sub_type'    =>  '',
                'lens_manufacturer'    =>  '',
                'base_curve'    =>  '',
                'lens_material'    =>  '',
                'lens_filter'    =>  '',
                'lens_colour'    =>  '',
                'coating_made'    =>  '',
                'coating_name'    =>  '',
                'coating_colour'    =>  '',
            ],
            'rx'    =>  [
                'distance'  =>  [
                    'right_sph' =>  $data->r_sphere,
                    'right_cyl' =>  $data->r_cylinder,
                    'right_axis' =>  $data->r_axis,
                    'right_oc' =>  '',
                    'left_sph' =>  $data->l_sphere,
                    'left_cyl' =>  $data->l_cylinder,
                    'left_axis' =>  $data->l_axis,
                    'left_oc' =>  '',
                ],
                'prism' =>  [
                    'right_prism1' => $data->r_prism1,
                    'right_base1'   =>  $data->r_base1,
                    'right_prism2' => $data->r_prism2,
                    'right_base2'   =>  $data->r_base2,
                    'left_prism1' => $data->l_prism1,
                    'left_base1'   =>  $data->l_base1,
                    'left_prism2' => $data->l_prism2,
                    'left_base2'   =>  $data->l_base2,
                ],
                'add_for_reading'   =>  [
                    'right_add' =>  $data->r_add,
                    'right_boc' =>  '',
                    'right_sh' =>  $data->r_seg_height,
                    'right_intermediate_strength' =>  '',
                    'right_boc' =>  '',
                    'left_add' =>  $data->l_add,
                    'left_boc' =>  '',
                    'left_sh' =>  $data->l_seg_height,
                    'left_intermediate_strength' =>  '',
                    'left_boc' =>  ''
                ],
                'extended_info'=>[
                    'right_pd'  =>  $data->r_distance_pd,
                    'right_intermediate_pd'  =>  '',
                    'right_npd'  =>  $data->r_near_pd,
                    'right_base_curve'  =>  $data->r_base_curve,
                    'left_pd'  =>  $data->l_distance_pd,
                    'left_intermediate_pd'  =>  '',
                    'left_npd'  =>  $data->l_near_pd,
                    'left_base_curve'  =>  $data->l_base_curve,
                ],
                'special_info'=>[
                    'pento' =>  '',
                    'wrap'  =>  '',
                    'tilt'  =>  ''
                ]
            ],
            'frame' =>  [
                'job_type'  =>  (strtolower($data->frame) == "enclosed") ? "Edged & Mounted (Frame to Come)" : ucfirst($data->frame),
                'frame_type'    =>  $data->frame_type,
                'invoice_order_no'  =>  '',
                'ft_other'  =>  '',
                'specify_thickness' =>  $data->safety_thickness != '' ? 'on' : '',
                'safety_stamp'  =>  $data->safety_stamp != "No" ? 'on' : '',
                'bevel_type'    =>  '',
                'edge_polish'   =>  $data->polish_edge != 0 ? 'on' : '',
                'frame_info'    =>  [
                    'brand' =>  $data->brand,
                    'model' =>  $data->model,
                    'colour' =>  $data->color,
                    'pof' =>  ''
                ],
                'frame_size'    =>  [
                    'a' =>  $data->frame_size_a,
                    'b' =>  $data->frame_size_b,
                    'ed' =>  $data->frame_size_ed,
                    'dbl' =>  $data->frame_size_dbl,
                    'shape' =>  '',
                    'ed_or_shape'   =>  'ed'
                ],
                'modify_shape'  =>  [
                    'a_increased_by'    =>  '',
                    'a_shape_options'   =>  $data->modify_shape_a,
                    'b_increased_by'    =>  '',
                    'b_shape_options'   =>  $data->modify_shape_b
                ],
                'specify_thickness' =>  [
                    'ct_or_et'  =>  !empty($data->thickness_ct_et) ? $data->thickness_ct_et.' mm' : ''
                ]
            ],
            'additional_items'  =>  [
                'add_clip_on'   =>  '',
                'add_repairs'   =>  '',
                'clip_on_list'  =>  '',
                'repair_description'    =>  ''
            ]
        ];
    }

    public function isInternalMP()
    {
        $ordRes = $this->getShowData();

        // Check if lens type contains 'Stock'
        if (strpos($ordRes['lens']['lens_type'], 'Single Vision Finished') !== false) {
            return 0;
        }
        $rxSphRight = $ordRes['distance']['right']['sph'];
        $rxSphLeft = $ordRes['distance']['left']['sph'];
        $rxCylRight = $ordRes['distance']['right']['cyl'];
        $rxCylLeft = $ordRes['distance']['left']['cyl'];
        $rxAxisRight = $ordRes['distance']['right']['axis'];
        $rxAxisLeft = $ordRes['distance']['left']['axis'];

        $rxSphRightFloat = floatval($rxSphRight);
        $rxCylRightFloat = floatval($rxCylRight);
        $rxSphLeftFloat = floatval($rxSphLeft);
        $rxCylLeftFloat = floatval($rxCylLeft);

        // Check if both eyes are zero
        if (($rxSphRight === '' || $rxSphRightFloat == 0) && ($rxCylRight === '' || $rxCylRightFloat == 0)) {
            return 0;
        }
        if (($rxSphLeft === '' || $rxSphLeftFloat == 0) && ($rxCylLeft === '' || $rxCylLeftFloat == 0)) {
            return 0;
        }
        // Fetch all relevant orders from the last 90 days
        $recentOrders = Order::join('rx_orders', 'orders.id', '=', 'rx_orders.order_id')
            ->where('orders.id', '!=', $this->order_id)
            ->where('orders.type','rx')
            ->where('orders.created_at', '>=', now()->subDays(90))
            ->get();

        $exists = 0;

        foreach ($recentOrders as $order) {
            $rxOrder = self::where('order_id',$order->order_id)->first();
            if (empty($rxOrder)) {
                continue;
            }
            $data = $rxOrder->getShowData();

            if (!$data || !isset($data['distance'])) {
                continue;
            }
            // print_r($data);exit;
            // Match JSON fields
            $distance = $data['distance'];
            if (
                $distance['right']['sph'] == $rxSphRight &&
                $distance['left']['sph'] == $rxSphLeft &&
                $distance['right']['cyl'] == $rxCylRight &&
                $distance['left']['cyl'] == $rxCylLeft &&
                $distance['right']['axis'] == $rxAxisRight &&
                $distance['left']['axis'] == $rxAxisLeft &&
                stripos($data['lens']['lens_type'], 'Single Vision Finished') === false
            ) {
                // Check if the order is within the last 90 days
                // $createdAt = isset($data['created_at']) ? new \DateTime($data['created_at']) : null;
                // if ($createdAt && (new \DateTime())->diff($createdAt)->days <= 90) {
                    // Additional patient prefix check if Cyl values are zero
                    if ($rxCylRightFloat == 0 && $rxCylLeftFloat == 0) {
                        $patientPrefix = substr($ordRes['name'], 0, 5);
                        if (substr($data['name'], 0, 5) !== $patientPrefix) {
                            continue;
                        }
                    }

                    $exists = $order->order_id;
                    break;
                // }
            }
        }

        return $exists;
    }
}