<?php

namespace App\Services;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;
use App\Models\Backend\Order;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class LabOrderApiService
{
    protected $client;
    // Developer mode keys
    protected $baseUrl = 'https://gatekeeper-staging.opticalonline.com/api/v2/';
    protected $jwtKey = '2b0VckQOOjzNBgznP7nK-g';
    protected $jwtSecret = 'X9KpDrm2lCwF93899deCbg';
    // Live mode key
    // protected $baseUrl = 'https://gatekeeper.opticalonline.com/api/v2/';
    // protected $jwtKey = 'D3Q-_jBp3LgRHyonJoPV8MuKzY-ezHreoeI6I5lyPQTf-9YxzJ12J-U3BGi7cnE_xJNfbtjKA-jhf6_f';
    // protected $jwtSecret = 'DAlPYB9-Vvb3j9lvQOyQI0HrgkPr90UUFNiUo5ondaOGweJi2-Sj6Bfrw4cBDQqIk4CcWAS4520L6UDo8rn2DxcV35KE4PVHnVZEo-GS4zxUBc-Seen-N9iv';

    protected $cacheKey = 'lab_order_auth_token';
    protected $cacheTTL = 1440; // 24 hours in minutes

    public function __construct()
    {
        $this->client = new Client([
            'base_uri' => $this->baseUrl,
            'timeout' => 30.0,
        ]);
    }

    /**
     * Get or refresh authentication token
     * @return string
     * @throws \Exception
     */
    public function getAuthToken()
    {
        if (Cache::has($this->cacheKey)) {
            return Cache::get($this->cacheKey);
        }

        try {
            $response = $this->client->post('auth_user', [
                'form_params' => [
                    'jwt_key' => $this->jwtKey,
                    'jwt_secret' => $this->jwtSecret
                ]
            ]);
            $bodyContent = $response->getBody()->getContents();
            $data = json_decode($bodyContent, true);
            if (!isset($data['auth_token'])) {
                throw new \Exception('Failed to retrieve auth token');
            }
            // Cache token for 24 hours
            Cache::put($this->cacheKey, $data['auth_token'], $this->cacheTTL);
            return $data['auth_token'];
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get Hash routing key
    */
    public function getHashRouting()
    {
        try {
            $token = $this->getAuthToken();
            $response = $this->client->get('operations/contract_available', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'Content-Type' => 'application/json',
                ],
            ]);
            return json_decode($response->getBody()->getContents(), true);

        } catch (\Exception $e) {
            \Log::error('Failed to push order to lab: ' . $e->getMessage());
            throw $e;
        }
    }

    public function canPushOrder(Order $order)
        {
            $config = getLabzillaAccountConfig();
            if (empty($config)) {
                \Log::channel('order_processing')->error("Config is empty");
                return false;
            }
            if (isset($config['status']) && $config['status'] == 'false') {
                \Log::channel('order_processing')->error("Status is false");
                return false;
            }

            \Log::channel('order_processing')->info("Order type: {$order->type}, Allowed types: " . json_encode($config['order_types']));
            if (isset($config['order_types']) && !in_array($order->type, $config['order_types'])) {
                \Log::channel('order_processing')->error("Order type {$order->type} not allowed");
                return false;
            }

            if (isset($config['user_accounts'])) {
                \Log::channel('order_processing')->info("User id: {$order->user->id}, Allowed accounts: " . json_encode($config['user_accounts']));
                if ($config['user_accounts'] == "all") {
                    \Log::channel('order_processing')->info("All user accounts allowed");
                    return true;
                }
                if (is_array($config['user_accounts'])) {
                    if (in_array("all", $config['user_accounts'])) {
                        \Log::channel('order_processing')->info("All user accounts allowed (array)");
                        return true;
                    }
                    if (in_array((string)$order->user->id, $config['user_accounts'], true)) {
                        \Log::channel('order_processing')->info("User id {$order->user->id} is allowed");
                        return true;
                    }
                    \Log::channel('order_processing')->error("User id {$order->user->id} not in allowed list");
                    return false;
                }
                \Log::channel('order_processing')->error("User accounts config is invalid");
                return false;
            }
            \Log::channel('order_processing')->info("No user accounts restriction");
            return true;
    }

    public function pushOrderToLab(array $orderData, Order $order, $manualClick)
    {
        if($manualClick != true) {
            if (!$this->canPushOrder($order)) {
                $response =  [
                    'success' => false,
                    'message' => 'Order is not eligible for pushing to the lab based on the current configuration.'
                ];
                Log::channel('order_processing')->error("Order is not eligible for pushing to the lab based on the current configuration. ID: {$order->id}");
                return $response;
            }
        }
        try {
            $token = $this->getAuthToken();
            Log::channel('order_processing')->info("Token Pass");
            $response = $this->client->post('orders/push_order_to_lab', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'Content-Type' => 'application/json',
                ],
                'json' => $orderData
            ]);

            return json_decode($response->getBody()->getContents(), true);

        } catch (\Exception $e) {
            \Log::error('Failed to push order to lab: ' . $e->getMessage());
            throw $e;
        }
    }
}