(function() {
    function init() {

        localStorage.removeItem('updatedElementsDta');
        localStorage.removeItem('existingElementsDta');
        let inputs = document.querySelectorAll('.common-form-control');
        inputs.forEach(input => {
            // Get or create change-log div
            let logDiv = input.nextElementSibling;
            if (!logDiv || !logDiv.classList.contains('change-log')) {
                logDiv = document.createElement('div');
                logDiv.classList.add('change-log');
                input.parentNode.insertBefore(logDiv, input.nextSibling);
            }

            // Add input formatting for phone/fax
            if (input.name === 'phone2' || input.name === 'fax') {
                // Prevent default behavior for arrow keys to maintain cursor position
                input.addEventListener('keydown', function(e) {
                    // Allow navigation keys
                    if ([37, 38, 39, 40].indexOf(e.keyCode) !== -1) {
                        return;
                    }
                    
                    // Allow backspace, delete, tab, etc.
                    if ([8, 9, 46].indexOf(e.keyCode) !== -1) {
                        return;
                    }
                    
                    // Allow Ctrl/Command + A, C, V, X
                    if ((e.ctrlKey || e.metaKey) && [65, 67, 86, 88].indexOf(e.keyCode) !== -1) {
                        return;
                    }
                    
                    // Prevent non-numeric input
                    if (e.keyCode < 48 || e.keyCode > 57) {
                        e.preventDefault();
                    }
                });

                input.addEventListener('input', function() {
                    formatPhoneInput(this);
                });
            }
            
            const handleInput = () => {
                // Hide change-log while user is typing
                logDiv.style.display = 'none';
                const key = input.name;
                const date = new Date();
                const hours = date.getHours();
                const minutes = date.getMinutes();
                const ampm = hours >= 12 ? 'pm' : 'am';
                const formattedHours = (hours % 12 || 12).toString().padStart(2, '0'); // Add leading zero
                const formattedMinutes = minutes.toString().padStart(2, '0'); // Add leading zero
                const monthName = date.toLocaleString('default', { month: 'short' });
                const day = date.getDate().toString().padStart(2, '0'); // Add leading zero
                const formattedDate = `${day}-${monthName}-${date.getFullYear()} ${formattedHours}:${formattedMinutes} ${ampm}`;
                
                const savedData = localStorage.getItem('existingElementsDta');
                const existingElementsDta = savedData ? JSON.parse(savedData) : {};
                
                let updatedElementsDta = JSON.parse(localStorage.getItem('updatedElementsDta')) || {};
                updatedElementsDta[key] = {
                    old: existingElementsDta[key] || "", 
                    new: input.value, 
                    date: formattedDate 
                };
                localStorage.setItem('updatedElementsDta', JSON.stringify(updatedElementsDta));
                
                // Update log div content but keep it hidden for now
                logDiv.innerHTML = '';
                const logEntry = document.createElement('p');
                logEntry.textContent = `Change to ${input.value} submitted on ${formattedDate}`;
                logDiv.appendChild(logEntry);
            };
            
            input.addEventListener('input', handleInput);
            const submitBtn = document.getElementById('submitBtn');
            if (submitBtn) {
                submitBtn.addEventListener('click', function () {
                });
            }
        });
        CURD('onload', null);
        scheduleRemoval();

        // Add this to your initialization code
        document.querySelectorAll('input[name="phone2"], input[name="fax"]').forEach(input => {
            input.addEventListener('paste', function(e) {
                e.preventDefault();
                const pastedText = (e.clipboardData || window.clipboardData).getData('text');
                const digits = pastedText.replace(/\D/g, '');
                
                // Remove leading 1 if present (we'll add it in formatting)
                const cleanDigits = digits.startsWith('1') ? digits.substring(1) : digits;
                
                // Insert the digits at current cursor position
                const startPos = this.selectionStart;
                const endPos = this.selectionEnd;
                const currentValue = this.value.replace(/\D/g, '');
                
                const newValue = currentValue.substring(0, startPos) + cleanDigits + currentValue.substring(endPos);
                this.value = newValue;
                
                // Trigger formatting
                formatPhoneInput(this);
                
                // Move cursor to end of pasted text
                const newCursorPos = startPos + cleanDigits.length;
                this.setSelectionRange(newCursorPos, newCursorPos);
            });
        });
    }

    function updateUI(requestData) {

        const inputs = document.querySelectorAll('.common-form-control');
        inputs.forEach(input => {
            const key = input.name;
            let logDiv = input.nextElementSibling;
            if (!logDiv || !logDiv.classList.contains('change-log')) {
                logDiv = document.createElement('div');
                logDiv.classList.add('change-log');
                input.parentNode.insertBefore(logDiv, input.nextSibling);
            }

            if (requestData.hasOwnProperty(key)) {
                logDiv.innerHTML = '';
                const logEntry = document.createElement('p');
                logEntry.innerHTML = `Change to <strong>${requestData[key].new}</strong> submitted on ${requestData[key].date}`;
                logDiv.appendChild(logEntry);
                input.value = requestData[key].old;
            } else {
                // console.warn(`No matching data found for input: ${key}`);
            }
        });
    }

    function CURD(requestType, requestData)
    {
        const userId = window.userId;
        const userData = { userId: userId, requestType: requestType, requestData: requestData };
        fetch('/request-change', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ userData })
        })
        .then(response => response.json()) 
        .then(data => {
            if(requestType == 'update') {
                $('.loader').hide();
                localStorage.removeItem('updatedElementsDta');
                localStorage.removeItem('existingElementsDta');
                document.getElementById('requestChangeBtn').style.display = 'inline';
                document.getElementById('cancelBtn').style.display = 'none';
                document.getElementById('submitBtn').style.display = 'none';
                const inputs = document.querySelectorAll('.common-form-control');
                inputs.forEach(input => {
                    input.setAttribute('readonly', true);
                });
            }
            const mergedRequestData = { ...requestData, ...data.data.requestData };
            updateUI(mergedRequestData);
            if (data.success) {
                $('.loader').hide();
                // Show all change logs only after successful submission
                document.querySelectorAll('.change-log').forEach(log => {
                    log.style.display = 'block';
                });
                toastr.success("Admin has been notified successfully.");
            } else if (data.success == '') {
                $('.loader').hide();
            } else {
                $('.loader').hide();
                toastr.error("Failed to notify admin. Please try again.");
            }
        })
        .catch(error => {
            $('.loader').hide();
            // console.error('Error:', error);
        });
    }

    function requestChange()
    {
        document.getElementById('requestChangeBtn').style.display = 'none';
        document.getElementById('cancelBtn').style.display = 'inline';
        document.getElementById('submitBtn').style.display = 'inline';
        let existingElementsDta = {};
        const inputs = document.querySelectorAll('.common-form-control');
        inputs.forEach(input => {
            input.removeAttribute('readonly');
            existingElementsDta[input.name] = input.value;
        });
        localStorage.setItem('existingElementsDta', JSON.stringify(existingElementsDta));
    }

    window.requestChange = requestChange;
    window.cancelChange = cancelChange;

    function cancelChange()
    {
        document.getElementById('requestChangeBtn').style.display = 'inline';
        document.getElementById('cancelBtn').style.display = 'none';
        document.getElementById('submitBtn').style.display = 'none';

        const inputs = document.querySelectorAll('.common-form-control');
        const existingElementsDta = JSON.parse(localStorage.getItem('existingElementsDta')) || {};
        const updatedElementsDta = JSON.parse(localStorage.getItem('updatedElementsDta')) || {};

        inputs.forEach(input => {
            input.setAttribute('readonly', true);
            input.value = existingElementsDta[input.name] || ''; // Reset to original value
            let logDiv = input.nextElementSibling;
            if (logDiv && logDiv.classList.contains('change-log') && updatedElementsDta.hasOwnProperty(input.name)) {
                logDiv.innerHTML = '';
            }
        });
        localStorage.removeItem('updatedElementsDta');
        localStorage.removeItem('existingElementsDta');
        location.reload();
    }

    window.submitChange = submitChange;

    function submitChange() {
        $('.loader').show();
        const updatedElementsDta = {};
        const inputs = document.querySelectorAll('.common-form-control');
        const existingElementsDta = JSON.parse(localStorage.getItem('existingElementsDta')) || {};

        // Format phone numbers before submission
        inputs.forEach(input => {
            if (input.name === 'phone2' || input.name === 'fax') {
                formatPhoneInput(input);
            }
        });

        inputs.forEach(input => {
            const key = input.name;
            const logDiv = input.nextElementSibling;
            if (logDiv && logDiv.classList.contains('change-log')) {
                const logEntry = logDiv.querySelector('p');
                if (logEntry) {
                    const logText = logEntry.textContent;
                    const newValueMatch = logText.match(/Change to (.*) submitted on/);
                    const dateMatch = logText.match(/submitted on (.*)/);
                    if (newValueMatch && dateMatch) {
                        updatedElementsDta[key] = {
                            old: existingElementsDta[key],
                            new: newValueMatch[1],
                            date: dateMatch[1]
                        };
                    }
                }
            }
        });
        if (Object.keys(updatedElementsDta).length === 0) {
            $('.loader').hide();
            return;
        }
        CURD('update', updatedElementsDta);
    }

    function scheduleRemoval()
    {
        const businessDays = 3;
        let mostRecentDate = null;
        const logs = document.querySelectorAll('.change-log');
        logs.forEach(log => {
            const logParagraph = log.querySelector('p');
            if (!logParagraph) return;
            let logText = logParagraph.innerHTML.trim();
            if (!logText) return;
            const match = logText.match(/submitted on (.+)/);
            if (match) {
                const logDateStr = match[1];
                const logDate = new Date(logDateStr);
                if (!mostRecentDate || logDate > mostRecentDate) {
                    mostRecentDate = logDate;
                }
            }
        });

        if (!mostRecentDate) return;
        let deleteDate = new Date(mostRecentDate);
        let addedDays = 0;
        while (addedDays < businessDays) {
            deleteDate.setDate(deleteDate.getDate() + 1);
            const dayOfWeek = deleteDate.getDay();
            if (dayOfWeek !== 0 && dayOfWeek !== 6) { // Skip weekends
                addedDays++;
            }
        }
        console.log('Scheduled deletion date:', deleteDate);
        const currentDate = new Date();
        const timeUntilDeletion = deleteDate - currentDate;

        if (timeUntilDeletion <= 0) {
            removeChangeLogs();
        } else {
            setTimeout(removeChangeLogs, timeUntilDeletion);
        }
    }

    function removeChangeLogs()
    {
        fetch('/delete-entry', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
        })
        .then(response => response.json())
        .then(data => {
            console.log('Removal successful');
            document.querySelectorAll('.change-log').forEach(log => log.remove());
            localStorage.removeItem('updatedElementsDta');
            localStorage.removeItem('existingElementsDta');
        })
        .catch(error => {
            document.querySelectorAll('.change-log').forEach(log => log.remove());
            localStorage.removeItem('updatedElementsDta');
            localStorage.removeItem('existingElementsDta');
        });
    }
    // Initial run (in case change-log is already populated)
    setTimeout(scheduleRemoval, 2000);


    function formatPhoneInput(input) {
        if (input.name !== 'phone2' && input.name !== 'fax') return;

        // Get cursor position before any changes
        const cursorPosition = input.selectionStart;
        const originalValue = input.value;

        // Extract digits only
        const digits = originalValue.replace(/\D/g, '');

        // Remove all characters if we don't have enough digits for a phone number
        if (digits.length === 0) {
            input.value = '';
            return;
        }

        // Always maintain the (+1) prefix
        let formattedValue = '';

        // Add the remaining digits with proper formatting
        const remainingDigits = digits.startsWith('1') ? digits.substring(1) : digits;
        
        if (remainingDigits.length > 0) {
            formattedValue += remainingDigits.substring(0, 3);
        }
        if (remainingDigits.length > 3) {
            formattedValue += '-' + remainingDigits.substring(3, 6);
        }
        if (remainingDigits.length > 6) {
            formattedValue += '-' + remainingDigits.substring(6, 10);
        }

        // Update the input value
        input.value = formattedValue;

        // Adjust cursor position to account for formatting characters
        let newCursorPosition = cursorPosition;
        
        // If we're typing at the end, move cursor to end
        if (cursorPosition >= originalValue.length) {
            newCursorPosition = formattedValue.length;
        }
        // Special handling for positions after formatting characters
        else if (cursorPosition === 4) { // After "(+1"
            newCursorPosition = 6; // Jump over ") "
        }
        else if (cursorPosition === 9) { // After first 3 digits
            newCursorPosition = 10; // Jump over "-"
        }
        else if (cursorPosition === 13) { // After second set
            newCursorPosition = 14; // Jump over "-"
        }

        input.setSelectionRange(newCursorPosition, newCursorPosition);
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();  // DOM is already ready—run now
    }
})();