
(function() {
    function init() {
        document.addEventListener('DOMContentLoaded', function () {
            document.addEventListener('click', function(event) {
                console.log('Clicked element:', event.target);
                const trElement = document.querySelector('tr.position-relative.active');
                const orderId = trElement ? trElement.getAttribute('data-id') : null;
                const orderType = trElement ? trElement.getAttribute('type') : null;

                // Update order model
                if (event.target.classList.contains('lab-order-model')) {
                    console.log('lab-order-model clicked');
                    event.preventDefault();
                    console.group('Labzilla order model')
                    console.log('orderId :', orderId);
                    console.log('orderType :', orderType);
                    console.groupEnd('end of labzilla order');
                    if (orderId) {
                        fetchOrderData(orderId, orderType, null, 'update')
                            .then((response) => {
                                showOrderSlider(response, orderType);
                            })
                            .catch(error => {
                                toastMessage('error', error.message || 'Failed to load order data');
                            });
                    } else {
                        toastMessage('error', 'Order ID not found');
                    }
                }
                // Submit order click function
                if (event.target.classList.contains('lab-order-submit')) {
                    event.preventDefault();
                    console.group('Labzilla order creation')
                    console.log('orderId :', orderId);
                    console.log('orderType :', orderType);
                    console.groupEnd('end of labzilla order');
                    let manualClick = true;
                    if (orderId) {
                        const originalText = event.target.textContent;
                        event.target.textContent = 'Submitting...';
                        event.target.disabled = true;
                        checkAuthToken()
                        .then((hashRouting) => {
                            return fetchOrderData(orderId, orderType, hashRouting, 'create');
                        })
                        .then(orderData => {
                            console.log('Order data to be sent to pushOrderToLab:', JSON.stringify(orderData, null, 2));
                            return pushOrderToLab(orderData, orderId, manualClick);
                        })
                        .then(response => {
                            if (response.success) {
                                toastMessage('success', 'Order submitted successfully!');
                            } else {
                                toastMessage('error', 'Failed to submit order: ' + response.message);
                            }
                        })
                        .catch(error => {
                            toastMessage('error', 'An error occurred while submitting the order: ' + error.message);
                        })
                        .finally(() => {
                            event.target.textContent = originalText;
                            event.target.disabled = false;
                        });
                    } else {
                        toastMessage('error', 'Order ID not found');
                    }
                }
            });
        });

        function showOrderSlider(response, orderType) {
            const orderData = response;
            if (!orderData) {
                toastMessage('error', 'Order data is not available.');
                return;
            }
            const displayData = orderType === 'ORDER_RX' && orderData.order ? { order: orderData.order } : orderData;
            const objectHtml = generateObjectHtml(displayData);

            // Create slider HTML with object display
            const sliderHTML = `
                <div class="order-slider-container">
                    <div class="order-slider-overlay"></div>
                    <div class="order-slider-panel">
                        <div class="order-slider-header">
                            <h4>Lab Zilla Order Data</h4>
                            <button class="btn btn-primary btn-sm request-json-btn">Request Json</button>
                            <button class="close-slider">&times;</button>
                        </div>
                        <div class="order-slider-content">
                            <div class="object-display">${objectHtml}</div>
                        </div>
                    </div>
                </div>
            `;

            // Find or create the order-list col-md-6 container
            let container = document.querySelector('.order-list.col-md-6');
            if (!container) {
                container = document.createElement('div');
                container.className = 'order-list col-md-6';
                document.body.appendChild(container);
            }

            // Add slider to the container
            container.insertAdjacentHTML('beforeend', sliderHTML);

            // Add event listener for request json button
            document.querySelector('.request-json-btn').addEventListener('click', function() {
                const trElement = document.querySelector('tr.position-relative.active');
                const orderId = trElement ? trElement.getAttribute('data-id') : null;
                const orderType = trElement ? trElement.getAttribute('type') : null;

                if (orderId) {
                    checkAuthToken()
                    .then((hashRouting) => {
                        return fetchOrderData(orderId, orderType, hashRouting, 'create');
                    })
                    .then(orderData => {
                        copyToClipboard(JSON.stringify(orderData, null, 2));
                    })
                    .catch(error => {
                        toastMessage('error', 'An error occurred while fetching JSON data: ' + error.message);
                    });
                } else {
                    toastMessage('error', 'Order ID not found');
                }
            });

            // Add event listeners for toggle buttons
            document.querySelectorAll('.toggle-button').forEach(button => {
                button.addEventListener('click', () => {
                    const ul = button.nextElementSibling;
                    if (ul.classList.contains('collapsed')) {
                        ul.classList.remove('collapsed');
                        button.classList.remove('collapsed');
                    } else {
                        ul.classList.add('collapsed');
                        button.classList.add('collapsed');
                    }
                });
            });

            // Add event listeners for close slider
            document.querySelector('.close-slider').addEventListener('click', closeSlider);
            document.querySelector('.order-slider-overlay')?.addEventListener('click', closeSlider);

            // Animate opening
            setTimeout(() => {
                document.querySelector('.order-slider-panel').classList.add('open');
                document.querySelector('.order-slider-overlay')?.classList.add('open');
            }, 10);
        }

        function generateObjectHtml(data, level = 0, isFieldObject = false) {
            if (data === null || typeof data !== 'object') {
                const valueClass = isFieldObject ? 'field-value' : 'object-value';
                return `<span class="${valueClass}">${data === null ? 'null' : data}</span>`;
            }

            const isArray = Array.isArray(data);
            const entries = isArray ? data.entries() : Object.entries(data);
            const isFieldArray = data.fields && Array.isArray(data.fields);
            let html = `<ul class="object-level object-level-${level} ${isArray ? 'array' : 'object'}">`;

            for (const [key, value] of entries) {
                const displayKey = isArray ? `[${key}]` : key;
                const keyClass = isFieldArray && ['our_field', 'labzilla_field', 'value'].includes(key) ? 'field-key' : 'object-key';
                const isNestedObject = value !== null && typeof value === 'object';
                const toggleButton = isNestedObject ? `<span class="toggle-button"></span>` : '';
                html += `
                    <li>
                        <span class="${keyClass}">${displayKey}:</span>
                        ${toggleButton}
                        ${generateObjectHtml(value, level + 1, isFieldArray && key === 'fields')}
                    </li>
                `;
            }

            html += '</ul>';
            return html;
        }

        function closeSlider() {
            const panel = document.querySelector('.order-slider-panel');
            const overlay = document.querySelector('.order-slider-overlay');

            if (panel && overlay) {
                panel.classList.remove('open');
                overlay.classList.remove('open');

                panel.addEventListener('transitionend', function handler() {
                    panel.parentElement.remove(); // Remove the entire order-slider-container
                    panel.removeEventListener('transitionend', handler);
                });
            }
        }

        function checkAuthToken() {
            return fetch('/admin/lab-order/check-auth', {
                method: 'GET',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content'),
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                credentials: 'same-origin'
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Auth token verified:', data.token);
                    return data.hash_routing;
                } else {
                    console.warn('Auth token check failed:', data.message);
                    toastMessage('error', 'Warning: Auth token check failed. The order submission might fail if authentication is required. Error: ' + data.message);
                    return Promise.resolve(null);
                }
            });
        }

        function fetchOrderData(orderId, orderType, hashRouting, type) {
            return fetch(`/admin/lab-order/data/${orderId}`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content'),
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify({ hash_routing: hashRouting, order_type: orderType, type: type })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    return data.data;
                } else {
                    throw new Error(data.message || 'Failed to fetch order data');
                }
            });
        }

        function pushOrderToLab(orderData, orderId, manualClick) {
            // return 'comment function for testing';
            return fetch('/admin/lab-order/push', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content'),
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify( {orderData: orderData, orderId: orderId, manualClick: manualClick })
            })
            .then(response => response.json());
        }

        function copyToClipboard(text) {
            let textArea = document.createElement("textarea");
            textArea.value = text;
            textArea.style.position = "fixed";
            textArea.style.left = "-9999px";
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            try {
                document.execCommand('copy');
                toastMessage('success', 'JSON copied to clipboard!');
            } catch (err) {
                toastMessage('error', 'Could not copy text: ' + err);
            }
            document.body.removeChild(textArea);
        }
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();  // DOM is already ready—run now
    }
})();